import { graphql } from 'react-relay/hooks';
import { PageviewMedium } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { GlowNotInterestedSvg } from 'tachyon-more-ui';
import type {
  ChannelInitialProps,
  ChannelPathParameters,
} from 'tachyon-page-utils';
import {
  channelIsFound,
  channelIsNotFoundServerside,
  channelPathGetInitialProps,
  defaultPageviewTracking,
  getCurrentUser,
} from 'tachyon-page-utils';
import { HorizontalNav } from 'tachyon-tv-nav';
import { CoreButtonType, Display, FlexDirection, Layout } from 'twitch-core-ui';
import { useNotInterested } from '../../../hooks';
import { RouteName, useTachyonRouter } from '../../../routing';
import {
  FocusableTextButton,
  SettingsPrompt,
  StarshotMain,
} from '../../common';
import { NotFoundError } from '../../errors';
import type { StarshotPage } from '../types';
import type { SettingsChannelNotInterested_QueryResponse } from './__generated__/SettingsChannelNotInterested_Query.graphql';

export type SettingsChannelNotInterestedProps = ChannelInitialProps &
  SettingsChannelNotInterested_QueryResponse;

export const SettingsChannelNotInterested: StarshotPage<
  ChannelInitialProps,
  SettingsChannelNotInterestedProps,
  ChannelPathParameters
> = ({ channel }) => {
  const { formatMessage } = useIntl();

  const { back, push } = useTachyonRouter();

  // TODO: determine correct values for the source* fields which assume
  // a user is declaring disinterest directly from a recommendation set which
  // is not necessarily the case in Starshot since they perform this action
  // from the channel profile page
  const notInterested = useNotInterested({
    itemID: channel?.id ?? '',
    itemType: 'CHANNEL',
    sourceItemPage: PageviewMedium.Browse,
    sourceItemRequestID: '',
    sourceItemTrackingID: '',
  });

  if (!channelIsFound(channel)) {
    return <NotFoundError />;
  }

  return (
    <StarshotMain>
      <SettingsPrompt
        subHeading={formatMessage(
          'We will no longer recommend content by {displayName} to you.',
          { displayName: channel.displayName },
          'SettingsChannelNotInterested',
        )}
        svg={<GlowNotInterestedSvg />}
        title={formatMessage('Are you sure?', 'SettingsChannelNotInterested')}
      >
        <HorizontalNav elementCount={2} focusIndex={0}>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
            <FocusableTextButton
              focusIndex={0}
              focusedType={CoreButtonType.Text}
              onClick={back}
            >
              {formatMessage('Go back', 'SettingsChannelNotInterested')}
            </FocusableTextButton>
            <FocusableTextButton
              focusIndex={1}
              focusedType={CoreButtonType.Destructive}
              onClick={() => {
                notInterested();
                push({ route: RouteName.Homepage });
              }}
            >
              {formatMessage("Yes, I'm sure", 'SettingsChannelNotInterested')}
            </FocusableTextButton>
          </Layout>
        </HorizontalNav>
      </SettingsPrompt>
    </StarshotMain>
  );
};

SettingsChannelNotInterested.currentUser = getCurrentUser;
SettingsChannelNotInterested.displayName = 'SettingsChannelNotInterested';
SettingsChannelNotInterested.getInitialProps = channelPathGetInitialProps;
SettingsChannelNotInterested.isNotFoundServerside = channelIsNotFoundServerside;
SettingsChannelNotInterested.navigationBehavior = () => ({
  displayNavMenu: true,
});
SettingsChannelNotInterested.pageviewTracking = defaultPageviewTracking;

SettingsChannelNotInterested.query = graphql`
  query SettingsChannelNotInterested_Query($login: String!) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    channel: user(login: $login) {
      id
      login
      displayName
    }
  }
`;
