import { fetchQuery, graphql } from 'react-relay/hooks';
import { getCurrentUserOnServer, logoutOnServer } from 'tachyon-auth-server';
import { logger } from 'tachyon-logger';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { initEnvironment } from 'tachyon-relay';
import type { StarshotRequestExtensions } from '../../../config';
import type { DomainList } from '../../../domainList';
import { dangerousDomainList } from '../../../domainList';
import { Redirect, RouteName } from '../../../routing';
import { SettingsMenu, StarshotMain } from '../../common';
import type { StarshotGetServerSideProps, StarshotPage } from '../types';
import type {
  SettingsDebugEnvironmentSelector_Query,
  SettingsDebugEnvironmentSelector_QueryResponse,
} from './__generated__/SettingsDebugEnvironmentSelector_Query.graphql';

type SettingsDebugEnvironmentSelectorServerSideProps = {
  domainList: DomainList;
};

export type SettingsDebugEnvironmentSelectorProps =
  SettingsDebugEnvironmentSelector_QueryResponse &
    SettingsDebugEnvironmentSelectorServerSideProps;

export const SettingsDebugEnvironmentSelector: StarshotPage<
  SettingsDebugEnvironmentSelectorServerSideProps,
  SettingsDebugEnvironmentSelectorProps
> = ({ domainList }) => {
  if (domainList.length === 0) {
    return <Redirect params={{ route: RouteName.Homepage }} />;
  }

  const domainMenuOptions = domainList.map((domain) => ({
    heading: domain.name ?? domain.url,
    interactive: {
      // istanbul ignore next: trivial
      onClick: () => (window.location.href = domain.url),
    },
  }));

  return (
    <StarshotMain>
      <SettingsMenu items={domainMenuOptions} />
    </StarshotMain>
  );
};

SettingsDebugEnvironmentSelector.currentUser = getCurrentUser;
SettingsDebugEnvironmentSelector.displayName =
  'SettingsDebugEnvironmentSelector';
SettingsDebugEnvironmentSelector.pageviewTracking = defaultPageviewTracking;
SettingsDebugEnvironmentSelector.navigationBehavior = () => ({
  displayBackButton: true,
  displayNavMenu: true,
});

// Detached to prevent a duplicate query during gIP
const SettingsDebugEnvironmentSelectorServerSideQuery = graphql`
  query SettingsDebugEnvironmentSelector_Query {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
  }
`;

// We're using getServerSideProps here so that the list of domains is only
// included in server side code. This will get tree shaken out on the
// application side thus avoiding leaking these domains to the public.
export const getServerSideProps: StarshotGetServerSideProps<
  SettingsDebugEnvironmentSelectorServerSideProps,
  {},
  StarshotRequestExtensions
> = async ({ req, res }) => {
  // allow environment override in pre-prod environments
  if (req.tachyon.appEnvironment !== 'production') {
    return { props: { domainList: dangerousDomainList } };
  }

  const { authorizationToken } = getCurrentUserOnServer(req, res);

  const relayEnvironment = initEnvironment({
    fetchQueryOpts: {
      authorization: {
        token: authorizationToken,
        unauthorizedHandler: () => {
          logoutOnServer(res);
        },
      },
    },
  });

  try {
    const data = await fetchQuery<SettingsDebugEnvironmentSelector_Query>(
      relayEnvironment,
      SettingsDebugEnvironmentSelectorServerSideQuery,
      {},
    ).toPromise();

    // we can expand this to include an allowlist of user ids as well
    if (data?.currentUser?.roles?.isStaff) {
      return { props: { domainList: dangerousDomainList } };
    }
  } catch {
    logger.log({
      category: 'SettingsDebugEnvironmentSelector',
      message: 'Failed GQL auth request',
      package: 'starshot',
    });
  }

  // once Next supports gSSP returning redirects, we should change the logic
  // here to return a homepage redirect when user is not staff and then remove
  // the redirect from the component itself
  return { props: { domainList: [] } };
};
