import { datatype, random } from 'faker';
import { fetchQuery } from 'react-relay/hooks';
import { logger } from 'tachyon-logger';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { dangerousDomainList } from '../../../domainList';
import { MovableList } from '../../common';
import { NotFound } from '../NotFound';
import { SettingsDebugEnvironmentSelector, getServerSideProps } from '.';

jest.mock('tachyon-logger', () => ({
  logger: {
    log: jest.fn(),
  },
}));
const mockLoggerLog = logger.log as jest.Mock;

jest.mock('tachyon-relay', () => ({
  initEnvironment: jest.fn(),
}));

jest.mock('tachyon-auth-server', () => ({
  getCurrentUserOnServer: () => ({ authorizationToken: 'foo' }),
}));

let mockFetchQueryValue: Promise<any> | undefined = undefined;
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  fetchQuery: jest.fn(() => ({ toPromise: () => mockFetchQueryValue })),
}));
const mockFetchQuery = fetchQuery as jest.Mock;

describe('SettingsEnvironment', () => {
  const mockRoles = { isStaff: false };
  const setup = createShallowWrapperFactory(
    SettingsDebugEnvironmentSelector,
    () => ({
      currentUser: {
        id: datatype.uuid(),
        login: random.alphaNumeric(),
        roles: mockRoles,
      },
      domainList: [{ url: 'test' }],
    }),
  );

  it('renders a 404 when user is not staff', () => {
    const { wrapper } = setup();

    expect(wrapper.find(NotFound)).not.toBeEmptyRender();
  });

  it('renders 404 when user is not logged in', () => {
    const { wrapper } = setup({ currentUser: null });

    expect(wrapper.find(NotFound)).not.toBeEmptyRender();
  });

  it('renders options when user is logged in and staff', () => {
    mockRoles.isStaff = true;
    const { wrapper } = setup();

    expect(wrapper.find(MovableList)).not.toBeEmptyRender();
  });

  describe(getServerSideProps, () => {
    beforeEach(() => {
      mockFetchQueryValue = undefined;
    });

    it('returns empty list if user is not staff and appEnvironment is production', async () => {
      mockFetchQueryValue = Promise.resolve({});
      const { props } = (await getServerSideProps({
        req: { tachyon: { appEnvironment: 'production' } },
      } as any)) as any;

      expect(mockFetchQuery).toHaveBeenCalled();
      expect(mockLoggerLog).not.toHaveBeenCalled();
      expect(props.domainList).toHaveLength(0);
    });

    it('returns empty list and logs if gql call fails and appEnvironment is production', async () => {
      mockFetchQueryValue = Promise.reject(new Error());
      const { props } = (await getServerSideProps({
        req: { tachyon: { appEnvironment: 'production' } },
      } as any)) as any;

      expect(mockFetchQuery).toHaveBeenCalled();
      expect(mockLoggerLog).toHaveBeenCalled();
      expect(props.domainList).toHaveLength(0);
    });

    it('returns populated list if user is staff and appEnvironment is production', async () => {
      mockFetchQueryValue = Promise.resolve({
        currentUser: { roles: { isStaff: true } },
      });
      const { props } = (await getServerSideProps({
        req: { tachyon: { appEnvironment: 'production' } },
      } as any)) as any;

      expect(mockFetchQuery).toHaveBeenCalled();
      expect(mockLoggerLog).not.toHaveBeenCalled();
      expect(props.domainList).toEqual(dangerousDomainList);
    });

    it('returns populated list and skips gql call if appEnvironment is not production', async () => {
      const { props } = (await getServerSideProps({
        req: { tachyon: { appEnvironment: 'staging' } },
      } as any)) as any;

      expect(mockFetchQuery).not.toHaveBeenCalled();
      expect(mockLoggerLog).not.toHaveBeenCalled();
      expect(props.domainList).toEqual(dangerousDomainList);
    });
  });
});
