import { useState } from 'react';
import { useStaticEnvironment } from 'tachyon-environment';
import type { UserExperimentMetadata } from 'tachyon-experiments';
import {
  ExperimentGroup,
  getExperimentOverridesQueryParams,
  getFriendlyExperimentName,
  useDangerousExperimentsConfig,
} from 'tachyon-experiments';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { SVGAsset } from 'twitch-core-ui';
import { EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY } from '../../../config';
import { Redirect, RouteName } from '../../../routing';
import type { SettingsMenuOption } from '../../common';
import { SettingsMenu, StarshotMain } from '../../common';
import type { StarshotPage } from '../types';

export const SettingsDebugExperimentManager: StarshotPage = () => {
  const { experimentGroupsForUser } = useDangerousExperimentsConfig();
  const [experiments, setExperiments] = useState(
    Object.values(experimentGroupsForUser).filter(
      (experiment): experiment is UserExperimentMetadata => !!experiment,
    ),
  );
  const {
    common: { appEnvironment },
  } = useStaticEnvironment();

  const toggleExperiment = (indexToToggle: number) => {
    const updatedExperiments = experiments.map((experiment, idx) => {
      if (idx !== indexToToggle) {
        return experiment;
      }
      return {
        ...experiment,
        group:
          experiment.group === ExperimentGroup.Treatment
            ? ExperimentGroup.Control
            : ExperimentGroup.Treatment,
      };
    });
    setExperiments(updatedExperiments);
  };

  const menuOptions: SettingsMenuOption[] = experiments.map(
    (experiment, experimentIndex) => ({
      heading: getFriendlyExperimentName(experiment),
      icon:
        experiment.group === ExperimentGroup.Treatment
          ? SVGAsset.Check
          : undefined,
      interactive: {
        onClick: () => {
          toggleExperiment(experimentIndex);
        },
      },
    }),
  );

  menuOptions.push({
    heading: 'Reload With Selected Experiments',
    icon: SVGAsset.Refresh,
    interactive: {
      onClick: () => {
        // Do a full window refresh in order for the app to properly re-initialize with the
        // experiment overrides.
        window.location.assign(
          `/?${EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY}=${getExperimentOverridesQueryParams(
            experiments,
          )}`,
        );
      },
    },
  });

  if (appEnvironment === 'production') {
    return <Redirect params={{ route: RouteName.Homepage }} />;
  }

  return (
    <StarshotMain>
      <SettingsMenu items={menuOptions} />
    </StarshotMain>
  );
};

SettingsDebugExperimentManager.currentUser = getCurrentUser;
SettingsDebugExperimentManager.displayName = 'SettingsDebugExperimentManager';
SettingsDebugExperimentManager.navigationBehavior = () => ({
  displayBackButton: true,
  displayNavMenu: true,
});
SettingsDebugExperimentManager.pageviewTracking = defaultPageviewTracking;
