import { graphql } from 'react-relay/hooks';
import { useCurrentUser } from 'tachyon-auth';
import { useIntl } from 'tachyon-intl';
import type { VodInitialProps, VodPathParameters } from 'tachyon-page-utils';
import {
  channelIsFound,
  defaultPageviewTracking,
  getCurrentUser,
  vodGetInitialProps,
  vodIsFound,
  vodIsNotFoundServerside,
} from 'tachyon-page-utils';
import { SVGAsset } from 'twitch-core-ui';
import {
  RouteName,
  createRedirectQuery,
  renderTachyonLink,
} from '../../../routing';
import type { SettingsMenuOption } from '../../common';
import { SettingsMenu, StarshotMain } from '../../common';
import { NotFoundError } from '../../errors';
import type { StarshotPage } from '../types';
import type { SettingsVod_QueryResponse } from './__generated__/SettingsVod_Query.graphql';

export type SettingsVodProps = SettingsVod_QueryResponse & VodInitialProps;

export const SettingsVod: StarshotPage<
  VodInitialProps,
  SettingsVodProps,
  VodPathParameters
> = ({ video }) => {
  const { formatMessage } = useIntl();
  const { loggedIn } = useCurrentUser();

  if (!vodIsFound(video) || !channelIsFound(video.owner)) {
    return <NotFoundError />;
  }

  const menuOptions: SettingsMenuOption[] = [
    {
      heading: formatMessage('Report Video', 'SettingsVod'),
      icon: SVGAsset.Edit,
      interactive: {
        linkTo: '/deferToRenderLink',
        renderLink: renderTachyonLink({
          route: RouteName.SettingsVodReport,
          routeParams: {
            id: video.id,
          },
        }),
      },
      subMenuIcon: true,
      subheading: formatMessage(
        'Is this video violating ToS? Tell us about it!',
        'SettingsVod',
      ),
    },
    {
      heading: formatMessage('Report Channel', 'SettingsVod'),
      icon: SVGAsset.Edit,
      interactive: {
        linkTo: '/deferToRenderLink',
        renderLink: renderTachyonLink({
          route: RouteName.SettingsChannelReport,
          routeParams: {
            login: video.owner.login,
          },
        }),
      },
      subMenuIcon: true,
      subheading: formatMessage(
        'Is this channel violating ToS? Tell us about it!',
        'SettingsVod',
      ),
    },
    {
      heading: formatMessage('Not interested in this channel?', 'SettingsVod'),
      icon: SVGAsset.NavCreative,
      interactive: {
        linkTo: '/deferToRenderLink',
        renderLink: loggedIn
          ? renderTachyonLink({
              route: RouteName.SettingsChannelNotInterested,
              routeParams: {
                login: video.owner.login,
              },
            })
          : renderTachyonLink({
              query: createRedirectQuery({
                route: RouteName.SettingsChannelNotInterested,
                routeParams: { login: video.owner.login },
              }),
              route: RouteName.Login,
            }),
      },
      subMenuIcon: true,
      subheading: formatMessage(
        "We won't recommend it to you anymore",
        'SettingsVod',
      ),
    },
  ];

  return (
    <StarshotMain>
      <SettingsMenu items={menuOptions} />
    </StarshotMain>
  );
};

SettingsVod.currentUser = getCurrentUser;
SettingsVod.displayName = 'SettingsVod';
SettingsVod.navigationBehavior = () => ({
  displayBackButton: true,
});
SettingsVod.pageviewTracking = defaultPageviewTracking;
SettingsVod.isNotFoundServerside = vodIsNotFoundServerside;
SettingsVod.getInitialProps = vodGetInitialProps;

SettingsVod.query = graphql`
  query SettingsVod_Query($videoId: ID) {
    currentUser {
      ...types_currentUser @relay(mask: false)
    }
    video: video(id: $videoId) {
      owner {
        id
        login
      }
      id
    }
  }
`;
