import { datatype, internet } from 'faker';
import { useCurrentUser } from 'tachyon-auth';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import {
  RouteName,
  createRedirectQuery,
  renderTachyonLink,
} from '../../../routing';
import { NotFoundError } from '../../errors';
import { SettingsVod } from '.';

jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useCurrentUser: jest.fn(() => ({ loggedIn: false })),
}));

jest.mock('../../../routing', () => ({
  ...jest.requireActual('../../../routing'),
  renderTachyonLink: jest.fn(() => () => <div />),
}));

const mockUseCurrentUser = useCurrentUser as jest.Mock;
const mockRenderTachyonLink = renderTachyonLink as jest.Mock;

describe(SettingsVod, () => {
  const setup = createShallowWrapperFactory(SettingsVod, () => ({
    currentUser: null,
    queryVariables: { videoId: datatype.uuid() },
    video: {
      id: datatype.uuid(),
      owner: {
        id: datatype.uuid(),
        login: internet.userName(),
      },
    },
  }));

  it('renders not found on missing video', () => {
    const { wrapper } = setup({ video: null });
    expect(wrapper.find(NotFoundError)).toExist();
  });

  describe('not interested button', () => {
    it('links to the SettingsVideoNotInterested page when logged in', () => {
      mockUseCurrentUser.mockImplementationOnce(() => ({
        loggedIn: true,
      }));
      const { props } = setup();

      expect(mockRenderTachyonLink).toHaveBeenCalledWith({
        route: RouteName.SettingsChannelNotInterested,
        routeParams: {
          login: props.video!.owner!.login,
        },
      });
    });

    it('links to login page when not logged in', () => {
      mockUseCurrentUser.mockImplementationOnce(() => ({
        loggedIn: false,
      }));
      const { props } = setup();

      expect(mockRenderTachyonLink).toHaveBeenCalledWith({
        query: createRedirectQuery({
          route: RouteName.SettingsChannelNotInterested,
          routeParams: {
            login: props.video!.owner!.login,
          },
        }),
        route: RouteName.Login,
      });
    });
  });
});
