import { usePlaybackTime } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { Color, CoreText, Icon, SVGAsset } from 'twitch-core-ui';
import { DarkThemeMap, StaticTokenMap } from 'twitch-core-ui-tokens';
import type { MutedSegmentPopup_muteInfo } from './__generated__/MutedSegmentPopup_muteInfo.graphql';

// istanbul ignore next: trivial
const ScInternalContainer = styled.div`
  background-color: ${DarkThemeMap[`color-background-base`]};
  border-bottom: 0.3rem solid ${StaticTokenMap['color-orange-12']};
  display: flex;
  flex-direction: row;
  padding: 1rem;
  width: 40rem;
`;

// istanbul ignore next: trivial
const ScWarningIconContainer = styled.div`
  color: ${DarkThemeMap[`color-background-warn`]};
  padding-right: 0.5rem;
`;

type MutedSegmentProps = {
  muteInfo: MutedSegmentPopup_muteInfo;
};

// This component and it's functionality will be reworked as a part of https://jira.xarth.tv/browse/EMP-4505 and https://jira.xarth.tv/browse/EMP-4504
export const MutedSegmentPopupBase: FC<MutedSegmentProps> = ({ muteInfo }) => {
  const currentTime = usePlaybackTime();
  const { formatMessage } = useIntl();

  let isMuted = false;
  const nodes = muteInfo.muteInfo?.mutedSegmentConnection?.nodes ?? [];

  for (const node of nodes) {
    if (
      currentTime >= node.offset &&
      currentTime <= node.offset + node.duration
    ) {
      isMuted = true;
      break;
    }
  }

  if (!isMuted) {
    return null;
  }

  return (
    <ScInternalContainer>
      <ScWarningIconContainer>
        <Icon asset={SVGAsset.NotificationWarning} />
      </ScWarningIconContainer>
      <div>
        <CoreText color={Color.Base}>
          {formatMessage('Copyrighted Content', 'MutedSegment')}
        </CoreText>
        <CoreText color={Color.Alt2}>
          {formatMessage(
            'Audio for portions of this video has been muted as it appears to contain certain copyrighted content owned or controlled by a third party',
            'MutedSegment',
          )}
        </CoreText>
      </div>
    </ScInternalContainer>
  );
};

MutedSegmentPopupBase.displayName = 'MutedSegmentPopupBase';

export const MutedSegmentPopup = createFragmentContainer(
  MutedSegmentPopupBase,
  {
    muteInfo: graphql`
      fragment MutedSegmentPopup_muteInfo on Video {
        muteInfo {
          mutedSegmentConnection {
            nodes {
              duration
              offset
            }
          }
        }
      }
    `,
  },
);

MutedSegmentPopup.displayName = 'MutedSegmentPopup';
