import { usePlaybackState } from 'pulsar';
import type { FC } from 'react';
import { useEffect, useRef } from 'react';
import type { SeekPreviewEntry } from '..';
import { useScrubControls } from '../../../../../player';

type SeekPreviewDisplayProps = {
  backgroundHeight: number;
  backgroundWidth: number;
  previewImages: Array<string>;
  seekPreviewEntry: SeekPreviewEntry;
};

// istanbul ignore next: trivial
export const SeekPreviewDisplay: FC<SeekPreviewDisplayProps> = ({
  backgroundHeight,
  backgroundWidth,
  previewImages,
  seekPreviewEntry,
}) => {
  const { subscribeToScrubbingPositionSeconds } = useScrubControls();
  const playbackState = usePlaybackState();
  const displayRef = useRef<HTMLDivElement | null>(null);

  const defaultStyles = `
    background-image: url("${previewImages[0]}");
    transform: translate3d(100%, 100%, 0);
    height: ${backgroundHeight}px;
    left: 0;
    position: absolute;
    top: 0;
    width: ${backgroundWidth}px;
    `;
  useEffect(() => {
    const updateDisplay = (nextPosition: number | undefined) => {
      if (displayRef.current) {
        // Don't render if we aren't scrubbing or if the player is seeking which will
        // be right after a scrub has been committed.
        if (nextPosition === undefined || playbackState === 'Buffering') {
          displayRef.current.style.cssText = defaultStyles;
          return;
        }

        const previewFrameNumber = Math.floor(
          nextPosition / seekPreviewEntry.interval,
        );
        // Ensure our timestamp wasn't too high
        if (previewFrameNumber > seekPreviewEntry.count) {
          displayRef.current.style.cssText = defaultStyles;
          return;
        }

        const index =
          previewFrameNumber % (seekPreviewEntry.rows * seekPreviewEntry.cols);

        displayRef.current.style.cssText = `
          background-image: url("${
            previewImages[
              Math.floor(
                previewFrameNumber /
                  (seekPreviewEntry.rows * seekPreviewEntry.cols),
              )
            ]
          }");
          transform: translate3d(
            -${(index % seekPreviewEntry.cols) * seekPreviewEntry.width}px,
            -${
              Math.floor(index / seekPreviewEntry.cols) *
              seekPreviewEntry.height
            }px,
            0);
          height: ${backgroundHeight}px;
          left: 0;
          position: absolute;
          top: 0;
          width: ${backgroundWidth}px;
        `;
      }
    };

    return subscribeToScrubbingPositionSeconds(updateDisplay);
  }, [
    backgroundHeight,
    backgroundWidth,
    defaultStyles,
    playbackState,
    previewImages,
    seekPreviewEntry.cols,
    seekPreviewEntry.count,
    seekPreviewEntry.height,
    seekPreviewEntry.interval,
    seekPreviewEntry.rows,
    seekPreviewEntry.width,
    subscribeToScrubbingPositionSeconds,
  ]);

  return (
    <div
      ref={displayRef}
      // eslint-disable-next-line react/forbid-dom-props
      style={{
        backgroundImage: `url("${previewImages[0]}")`,
        height: backgroundHeight,
        left: 0,
        position: 'absolute',
        top: 0,
        transform: `translate3d(100%, 100%, 0)`,
        width: backgroundWidth,
      }}
    />
  );
};

SeekPreviewDisplay.displayName = 'SeekPreviewDisplay';
