import type { FC } from 'react';
import { useCallback } from 'react';
import styled from 'styled-components';
import { fetchData, useAsync } from 'tachyon-utils';
import { Layout, Overflow, Position } from 'twitch-core-ui';
import { SeekPreviewDisplay } from './SeekPreviewDisplay';

export type SeekPreviewEntry = {
  cols: number;
  count: number;
  height: number;
  images: Array<string>;
  interval: number;
  quality: 'high' | 'low';
  rows: number;
  width: number;
};

export type SeekPreviewResponse = Array<SeekPreviewEntry>;

export type SeekPreviewProps = {
  seekPreviewsURL: string;
};

// istanbul ignore next: trivial
const ScPreviewContainer = styled(Layout)<{
  entryHeight: number;
  entryWidth: number;
}>`
  height: ${({ entryHeight }) => entryHeight}px;
  left: 50%;
  top: -25%;
  transform: translate(-50%, -50%) scale(2);
  transform-origin: bottom;
  width: ${({ entryWidth }) => entryWidth}px;
`;

// istanbul ignore next: trivial
const ScPreviewScopeLayout = styled(Layout)<{
  $backgroundWidth: number;
}>`
  ${({ $backgroundWidth }) => `background-size: ${$backgroundWidth}px;`}
`;

export const SeekPreview: FC<SeekPreviewProps> = ({ seekPreviewsURL }) => {
  const fetchPreviews = useCallback(() => {
    return fetchData<SeekPreviewResponse>(seekPreviewsURL);
  }, [seekPreviewsURL]);

  const { status, value } = useAsync<SeekPreviewResponse>(fetchPreviews);

  if (status !== 'fulfilled' || !value) {
    return null;
  }

  const highQualityEntry = value.find(({ quality }) => quality === 'high');

  if (!highQualityEntry) {
    return null;
  }

  const baseUrl = seekPreviewsURL.substring(
    0,
    seekPreviewsURL.lastIndexOf('/') + 1,
  );
  const previewImageUrls: Array<string> = [];

  // This data structure exists to cache the downloaded images
  // so that the app doesn't have to load the images on the fly
  // as we scrub and display separate sprite sheets
  const imageCacheArray: Array<HTMLImageElement> = [];

  highQualityEntry.images.forEach((imageUrlSuffix) => {
    const imageUrl = baseUrl + imageUrlSuffix;
    const newImage = new Image();
    newImage.src = imageUrl;
    imageCacheArray.push(newImage);
    previewImageUrls.push(imageUrl);
  });

  const backgroundWidth = highQualityEntry.cols * highQualityEntry.width;
  const backgroundHeight = highQualityEntry.height * highQualityEntry.rows;

  return (
    <ScPreviewContainer
      entryHeight={highQualityEntry.height}
      entryWidth={highQualityEntry.width}
      overflow={Overflow.Hidden}
      position={Position.Absolute}
    >
      <ScPreviewScopeLayout
        $backgroundWidth={backgroundWidth}
        fullHeight
        position={Position.Relative}
      >
        <SeekPreviewDisplay
          backgroundHeight={backgroundHeight}
          backgroundWidth={backgroundWidth}
          previewImages={previewImageUrls}
          seekPreviewEntry={highQualityEntry}
        />
      </ScPreviewScopeLayout>
    </ScPreviewContainer>
  );
};

SeekPreview.displayName = 'SeekPreview';
