import { datatype, lorem } from 'faker';
import { usePlaybackState } from 'pulsar';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { useAsync } from 'tachyon-utils';
import { useScrubControls } from '../../../../player';
import type { SeekPreviewResponse } from '.';
import { SeekPreview } from '.';

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  fetchData: jest.fn(),
  useAsync: jest.fn(),
}));

jest.mock('../../../../player', () => ({
  ...jest.requireActual('../../../../player'),
  useScrubControls: jest.fn(),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlaybackState: jest.fn(),
}));

const mockUseAsync = useAsync as jest.Mock;
const mockUseScrubControls = useScrubControls as jest.Mock;
const mockUsePlaybackState = usePlaybackState as jest.Mock;

const defaultScrubbingPositionSeconds = 1000;

type entryProps = {
  count?: number;
  quality?: 'high' | 'low';
};

function getMockSeekPreviewResponse({
  quality = 'high',
  count = defaultScrubbingPositionSeconds * 2,
}: entryProps): SeekPreviewResponse {
  return [
    {
      cols: datatype.number(),
      count,
      height: datatype.number(),
      images: [lorem.word()],
      interval: 1,
      quality,
      rows: datatype.number(),
      width: datatype.number(),
    },
  ];
}

describe(SeekPreview, () => {
  mockUseAsync.mockReturnValue({
    status: 'fulfilled',
    value: getMockSeekPreviewResponse({}),
  });

  mockUseScrubControls.mockReturnValue({
    scrubbingPositionSeconds: defaultScrubbingPositionSeconds,
  });

  mockUsePlaybackState.mockReturnValue('Playing');
  const setup = createShallowWrapperFactory(SeekPreview, () => ({
    seekPreviewsURL: 'tv.twitch.tv',
  }));

  it('renders when all required information is provided', () => {
    mockUseAsync.mockReturnValueOnce({
      status: 'fulfilled',
      value: getMockSeekPreviewResponse({}),
    });

    const { wrapper } = setup();

    expect(wrapper).not.toBeEmptyRender();
  });

  it('no renders when fetch is not fulfulled', () => {
    mockUseAsync.mockReturnValueOnce({
      status: 'none',
      value: getMockSeekPreviewResponse({}),
    });
    const { wrapper } = setup();

    expect(wrapper).toBeEmptyRender();
  });

  it('no renders when value is undefined', () => {
    mockUseAsync.mockReturnValueOnce({
      status: 'fulfilled',
      value: undefined,
    });

    const { wrapper } = setup();

    expect(wrapper).toBeEmptyRender();
  });

  it('no renders when there is no high quality previews', () => {
    mockUseAsync.mockReturnValueOnce({
      status: 'fulfilled',
      value: getMockSeekPreviewResponse({ quality: 'low' }),
    });

    const { wrapper } = setup();

    expect(wrapper).toBeEmptyRender();
  });
});
