import { useClosedCaptions, usePlaybackQuality } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { HorizontalNav } from 'tachyon-tv-nav';
import { AlignItems, Display, JustifyContent, Layout } from 'twitch-core-ui';
import { useFollow } from '../../../../../hooks';
import { RouteName } from '../../../../../routing';
import {
  FocusableFollowButton,
  PlaybackInfo,
  PlaybackOverlayItem,
} from '../../../../common';
import { ClosedCaptionsButton, QualitySelectButton } from '../../../../player';
import { VodSettingsButton } from './VodSettingsButton';
import type { VodBottomControls_video } from './__generated__/VodBottomControls_video.graphql';

export type BottomControlsProps = {
  focusIndex: number;
  video: VodBottomControls_video;
};

export const BottomControlsBase: FC<BottomControlsProps> = ({
  focusIndex,
  video,
}) => {
  const { following } = useFollow(video.owner);
  const {
    available: captionsAvailable,
    enabled: captionsEnabled,
    toggle: toggleCaptions,
  } = useClosedCaptions();

  const qualitySettings = usePlaybackQuality();

  const vodInfo = video.owner && (
    <PlaybackInfo
      channel={video.owner}
      focusIndex={0}
      gameDisplayName={video.game?.displayName}
      isLive={false}
      recordedAt={video.recordedAt}
      title={video.title}
      viewCount={video.viewCount ?? 0}
    />
  );

  const vodInfoFocusOffset = vodInfo ? 1 : 0;

  // Order matters for each of the buttons below
  const buttons = [
    <PlaybackOverlayItem key="settings">
      <VodSettingsButton focusIndex={vodInfoFocusOffset} id={video.id} />
    </PlaybackOverlayItem>,
  ];

  // Only show the follow button if the channel isn't already followed to
  // avoid accidental unfollows.
  // This UX will be improved as part of: https://jira.xarth.tv/browse/EMP-3714
  if (!following && video.owner) {
    buttons.push(
      <PlaybackOverlayItem key="follow">
        <FocusableFollowButton
          channel={video.owner}
          focusIndex={buttons.length + vodInfoFocusOffset}
          iconOnly
          route={RouteName.Channel}
        />
      </PlaybackOverlayItem>,
    );
  }

  if (captionsAvailable) {
    buttons.push(
      <PlaybackOverlayItem key="captions">
        <ClosedCaptionsButton
          captionsEnabled={captionsEnabled}
          focusIndex={buttons.length + vodInfoFocusOffset}
          onClick={toggleCaptions}
        />
      </PlaybackOverlayItem>,
    );
  }

  // Render this proactively even if the player hasn't initialized and qualities aren't yet
  // available for two reasons:
  // 1) Avoid a jarring UI update as a result of this being rendered after all of other UI has finished loading
  // 2) In a partial success case, there otherwise might not be anything to render
  //    in the bottom controls nav which can create a nav error state.
  buttons.push(
    <QualitySelectButton
      focusIndex={buttons.length + vodInfoFocusOffset}
      key="playback-quality"
      qualitySettings={qualitySettings}
    />,
  );

  // profile link + buttons
  const elementCount = vodInfoFocusOffset + buttons.length;

  // Ensure that VOD controls remain pushed to the right edge of the
  // layout even if there was a data response issue and the VOD info component
  // isn't rendered.
  const justify = vodInfo ? JustifyContent.Between : JustifyContent.End;

  return (
    <HorizontalNav elementCount={elementCount} focusIndex={focusIndex}>
      <Layout display={Display.Flex} fullWidth justifyContent={justify}>
        {vodInfo}
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexShrink={0}
        >
          {buttons}
        </Layout>
      </Layout>
    </HorizontalNav>
  );
};

BottomControlsBase.displayName = 'BottomControlsBase';

export const VodBottomControls = createFragmentContainer(BottomControlsBase, {
  video: graphql`
    fragment VodBottomControls_video on Video {
      owner {
        ...useFollow_targetChannel @relay(mask: false)
        ...FocusableFollowButton_channel
        login
        ...PlaybackInfo_channel
      }
      id
      recordedAt
      viewCount
      title
      game {
        displayName
      }
    }
  `,
});

VodBottomControls.displayName = 'VodBottomControls';
