import type { FC } from 'react';
import { useMemo } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { VerticalNav } from 'tachyon-tv-nav';
import { Layout } from 'twitch-core-ui';
import { useUpdateVideoPlaybackPosition } from '../../../../hooks';
import { PlayingUILayout } from '../../../common';
import {
  HideablePlayPauseButton,
  MediaSession,
  PlaybackScrubBar,
  ScrubControlsRoot,
  VodPlayerWrapper,
} from '../../../player';
import type { VodPlayerWrapperProps } from '../../../player';
import { MutedSegmentPopup } from './MutedSegmentPopup';
import { VodBottomControls } from './VodBottomControls';
import type { VodPlayingUI_currentUser } from './__generated__/VodPlayingUI_currentUser.graphql';
import type { VodPlayingUI_video } from './__generated__/VodPlayingUI_video.graphql';

type VodPlayingUIProps = Pick<VodPlayerWrapperProps, 'vodToken'> & {
  currentUser: VodPlayingUI_currentUser | null;
  focusIndex: number;
  video: VodPlayingUI_video;
};

export const VodPlayingUIBase: FC<VodPlayingUIProps> = ({
  currentUser,
  focusIndex,
  video,
  vodToken,
}) => {
  useUpdateVideoPlaybackPosition({
    userID: currentUser?.id,
    videoID: video.id,
    videoType: 'VOD',
  });

  // This is memoized because it's an expensive tree to re-render and it will
  // get thrashed by usePlaybackState as we seek, play, idle, and buffer
  return useMemo(
    () => (
      <VerticalNav
        elementCount={3}
        focusIndex={focusIndex}
        initialChildFocusIndex={1}
        takeFocusOnFirstRender
      >
        <PlayingUILayout
          channel={video.owner}
          live={false}
          overlayChildren={<HideablePlayPauseButton />}
          player={
            <VodPlayerWrapper
              autoPlay
              currentUser={currentUser}
              video={video}
              vodToken={vodToken}
            />
          }
        >
          <PlayingUILayout.BackButton focusIndex={0} />
          <PlayingUILayout.TopRight>
            <MutedSegmentPopup muteInfo={video} />
          </PlayingUILayout.TopRight>
          <PlayingUILayout.Controls>
            {/*
            Commenting out seek preview due to perf issues on LG televisions
            This functionality will be attempted to be reinstated with
            https://jira.xarth.tv/browse/EMP-4468
            {video.seekPreviewsURL && (
              <SeekPreview seekPreviewsURL={video.seekPreviewsURL} />
            )}
            */}
            <MediaSession />
            <Layout padding={{ bottom: 1 }}>
              <PlaybackScrubBar focusIndex={1} />
            </Layout>
            <VodBottomControls focusIndex={2} video={video} />
          </PlayingUILayout.Controls>
        </PlayingUILayout>
      </VerticalNav>
    ),
    [currentUser, focusIndex, video, vodToken],
  );
};

VodPlayingUIBase.displayName = 'VodPlayingUIBase';

// The pattern of wrapping a base component and passing props through is useful
// for having contexts available in the child component but also is a stable
// place for contexts to be instantiated. `VodPlayingUIBase` rerenders fairly
// frequently and contexts can be expensive to rerender.
// istanbul ignore next: trivial
const VodPlayingUIWrapper: FC<VodPlayingUIProps> = (props) => (
  <ScrubControlsRoot>
    <VodPlayingUIBase {...props} />
  </ScrubControlsRoot>
);

VodPlayingUIWrapper.displayName = 'VodPlayingUIWrapper';

export const VodPlayingUI = createFragmentContainer(VodPlayingUIWrapper, {
  currentUser: graphql`
    fragment VodPlayingUI_currentUser on User {
      ...VodPlayerBase_currentUser
      ...types_currentUser @relay(mask: false)
    }
  `,
  video: graphql`
    fragment VodPlayingUI_video on Video {
      ...VodBottomControls_video
      ...VodPlayerBase_video
      ...MutedSegmentPopup_muteInfo
      id
      owner {
        displayName
        login
        roles {
          isAffiliate
          isPartner
        }
      }
      seekPreviewsURL
    }
  `,
});
