import {
  PlayerControllerRoot,
  PlayerType,
  VodPlaybackRestrictionsRoot,
} from 'pulsar';
import { graphql } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import type {
  VodInitialProps as DefaultVodInitialProps,
  VodPathParameters,
} from 'tachyon-page-utils';
import {
  getCurrentUser,
  vodGetInitialProps,
  vodIsFound,
  vodIsNotFoundServerside,
  vodPageviewTracking,
} from 'tachyon-page-utils';
import { convertToUnsafeID } from 'tachyon-relay';
import { isBrowser } from 'tachyon-utils';
import { Redirect, RouteName } from '../../../routing';
import { StarshotMain } from '../../common';
import { NotFoundError } from '../../errors';
import { VodRestriction } from '../../player';
import type { StarshotPage } from '../types';
import { VodPlayingUI } from './VodPlayingUI';
import type {
  Vod_QueryResponse,
  Vod_QueryVariables,
} from './__generated__/Vod_Query.graphql';

export type VodInitialProps = DefaultVodInitialProps<Vod_QueryVariables>;

export type VodProps = Vod_QueryResponse & VodInitialProps;

export const Vod: StarshotPage<VodInitialProps, VodProps, VodPathParameters> =
  ({ currentUser, queryVariables, video, ...vodToken }) => {
    const {
      common: { platform },
    } = useStaticEnvironment();

    if (!vodIsFound(video)) {
      return <NotFoundError />;
    }

    if (platform === Platform.Switch) {
      return (
        <Redirect
          params={{
            route: RouteName.VodInterstitial,
            routeParams: { id: convertToUnsafeID(video.id) },
          }}
          replace
        />
      );
    }

    return (
      <StarshotMain>
        <PlayerControllerRoot>
          <VodPlaybackRestrictionsRoot video={video}>
            <VodRestriction focusIndex={0} video={video}>
              <VodPlayingUI
                currentUser={currentUser}
                focusIndex={0}
                video={video}
                vodToken={vodToken}
              />
            </VodRestriction>
          </VodPlaybackRestrictionsRoot>
        </PlayerControllerRoot>
      </StarshotMain>
    );
  };

Vod.currentUser = getCurrentUser;
Vod.displayName = 'Vod';
Vod.getInitialProps = (context) =>
  vodGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Pulsar,
    skipPlayToken: !isBrowser(),
  });
Vod.isNotFoundServerside = vodIsNotFoundServerside;
Vod.navigationBehavior = () => ({});
Vod.pageviewTracking = vodPageviewTracking;

Vod.query = graphql`
  query Vod_Query(
    $videoId: ID!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    video(id: $videoId) {
      ...VodPlayingUI_video
      ...VodRestriction_video @relay(mask: false)
      id
      broadcastType
      game {
        name
      }
      owner {
        id
        login
        stream {
          id
        }
      }
    }
    currentUser {
      ...types_currentUser @relay(mask: false)
      ...VodPlayingUI_currentUser
      id
      hasTurbo
    }
    ...VodPlayerWrapper_token
  }
`;
