import {
  PlayerControllerRoot,
  PlayerType,
  VodPlaybackRestrictionsRoot,
} from 'pulsar';
import { graphql } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { useIntl } from 'tachyon-intl';
import { StreamType, getStreamType } from 'tachyon-more-ui';
import type {
  VodInitialProps as DefaultVodInitialProps,
  VodPathParameters,
} from 'tachyon-page-utils';
import {
  getCurrentUser,
  vodGetInitialProps,
  vodIsFound,
  vodPageviewTracking,
} from 'tachyon-page-utils';
import { isBrowser } from 'tachyon-utils';
import {
  useInterstitialPlayAttempt,
  useIsContentPlayableOnSwitch,
  useStarshotNotification,
} from '../../../hooks';
import { Redirect, RouteName } from '../../../routing';
import {
  InterstitialLayout,
  InterstitialVideoDetails,
  StarshotMain,
} from '../../common';
import { NotFoundError } from '../../errors';
import { VodPlayerWrapper, VodRestriction } from '../../player';
import type { Vod_QueryVariables } from '../Vod/__generated__/Vod_Query.graphql';
import type { StarshotPage } from '../types';
import type { VodInterstitial_QueryResponse } from './__generated__/VodInterstitial_Query.graphql';

export type VodInterstitialInitialProps =
  DefaultVodInitialProps<Vod_QueryVariables>;

export type VodInterstitialProps = VodInterstitial_QueryResponse &
  VodInterstitialInitialProps;

export const VodInterstitial: StarshotPage<
  VodInterstitialInitialProps,
  VodInterstitialProps,
  VodPathParameters
> = ({ currentUser, queryVariables, video, ...vodToken }) => {
  const { attemptedPlay, resetAttemptedPlay, triggerAttemptedPlay } =
    useInterstitialPlayAttempt();

  const { publish } = useStarshotNotification();
  const { formatMessage } = useIntl();
  const {
    common: { platform },
  } = useStaticEnvironment();
  const { isContentPlayable, setManifestUrl } = useIsContentPlayableOnSwitch();

  if (!vodIsFound(video)) {
    return <NotFoundError />;
  }

  if (platform !== Platform.Switch) {
    return (
      <Redirect
        params={{
          route: RouteName.Vod,
          routeParams: { id: queryVariables.videoId },
        }}
        replace
      />
    );
  }

  const streamType = getStreamType({
    broadcaster: video.owner,
    type: video.owner?.stream?.type ?? null,
  });

  return (
    <StarshotMain>
      <PlayerControllerRoot>
        <VodPlaybackRestrictionsRoot video={video}>
          <VodRestriction
            focusIndex={0}
            onMutedSegmentsDetected={() => {
              publish({
                autoCloseMs: 5000,
                meta: {
                  displayType: 'in-feature-notification',
                  message: {
                    description: formatMessage(
                      'Audio for portions of this video has been muted as it appears to contain certain copyrighted content owned or controlled by a third party',
                      'VodInterstitial',
                    ),
                    title: formatMessage(
                      'Copyrighted Content',
                      'VodInterstitial',
                    ),
                  },
                },
                type: 'warning',
              });
            }}
            passThrough={!attemptedPlay}
            video={video}
          >
            <InterstitialLayout
              attemptedPlay={attemptedPlay}
              channel={video.owner}
              contentType="vod"
              isChannelLive={streamType === StreamType.Live}
              isContentPlayable={isContentPlayable}
              player={
                <VodPlayerWrapper
                  autoPlay={false}
                  currentUser={currentUser}
                  onManifestUrlCreated={setManifestUrl}
                  video={video}
                  vodToken={vodToken}
                />
              }
              resetAttemptedPlay={resetAttemptedPlay}
              triggerPlayAttempt={triggerAttemptedPlay}
            >
              <InterstitialVideoDetails
                categoryImageURL={video.game?.boxArtURL}
                categoryName={video.game?.displayName}
                createdAt={video.createdAt}
                videoDuration={video.lengthSeconds}
                videoTitle={video.title}
              />
            </InterstitialLayout>
          </VodRestriction>
        </VodPlaybackRestrictionsRoot>
      </PlayerControllerRoot>
    </StarshotMain>
  );
};

VodInterstitial.currentUser = getCurrentUser;
VodInterstitial.displayName = 'VodInterstitial';
VodInterstitial.navigationBehavior = () => ({ displayNavMenu: true });
VodInterstitial.pageviewTracking = vodPageviewTracking;
VodInterstitial.getInitialProps = (context) =>
  vodGetInitialProps(context, {
    platform: context.platform,
    playerType: PlayerType.Pulsar,
    skipPlayToken: !isBrowser(),
  });
VodInterstitial.query = graphql`
  query VodInterstitial_Query(
    $videoId: ID!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    video(id: $videoId) {
      ...VodPlayerBase_video
      ...VodPlayingUI_video
      ...VodRestriction_video @relay(mask: false)
      id
      broadcastType
      createdAt
      game {
        boxArtURL
        displayName
        name
      }
      lengthSeconds
      owner {
        ...InterstitialLayout_channel
        id
        login
        hosting {
          id
        }
        stream {
          id
          type
        }
      }
      title
    }
    currentUser {
      ...VodPlayerBase_currentUser
      ...types_currentUser @relay(mask: false)
    }
    ...VodPlayerWrapper_token
  }
`;
