import { act } from '@testing-library/react-hooks';
import { datatype, date, image, lorem } from 'faker';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { useStarshotNotification } from '../../../hooks';
import { Redirect } from '../../../routing';
import { NotFoundError } from '../../errors';
import { VodRestriction } from '../../player';
import type { VodInterstitialProps } from '.';
import { VodInterstitial } from '.';

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));

jest.mock('../../../hooks', () => ({
  ...jest.requireActual('../../../hooks'),
  useStarshotNotification: jest.fn(),
}));

const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;
const mockUseStarshotNotification = useStarshotNotification as jest.Mock;
const mockPublish = jest.fn();

const mockProps: VodInterstitialProps = {
  ' $fragmentRefs': { VodPlayerWrapper_token: true },
  currentUser: null,
  queryVariables: {
    platform: Platform.StarshotDev,
    playerType: 'pulsar',
    skipPlayToken: true,
    videoId: datatype.uuid(),
  },
  video: {
    ' $fragmentRefs': {
      VodPlayerBase_video: true,
      VodPlayingUI_video: true,
    },
    broadcastType: null,
    createdAt: date.past().toISOString(),
    game: {
      boxArtURL: image.imageUrl(),
      displayName: lorem.words(5),
      name: lorem.words(5),
    },
    id: datatype.uuid(),
    lengthSeconds: datatype.number(500),
    muteInfo: null,
    owner: {
      ' $fragmentRefs': {
        InterstitialLayout_channel: true as const,
      },
      broadcastSettings: {
        isMature: false,
      },
      displayName: lorem.word(),
      hosting: {
        id: datatype.uuid(),
      },
      id: datatype.uuid(),
      login: 'therealderekt',
      stream: null,
      subscriptionProducts: [
        {
          displayName: 'therealderekt',
          hasSubonlyVideoArchive: false,
        },
      ],
    },
    resourceRestriction: null,
    self: {
      isRestricted: false,
    },
    title: lorem.words(5),
  },
};

describe(VodInterstitial, () => {
  const setup = createShallowWrapperFactory(VodInterstitial, () => mockProps);

  beforeEach(() => {
    mockUseStaticEnvironment.mockReturnValue({
      common: { platform: Platform.Switch },
    });
    mockPublish.mockReset();
    mockUseStarshotNotification.mockReturnValue({ publish: mockPublish });
  });

  it('renders the not found UX when Vod is not found', () => {
    const { wrapper } = setup({ video: null });
    expect(wrapper.find(NotFoundError)).toExist();
  });

  it('redirects to the Homepage if platform is not Switch', () => {
    mockUseStaticEnvironment.mockReturnValue({
      common: { platform: Platform.LG },
    });

    const { wrapper } = setup();
    expect(wrapper.find(Redirect)).toExist();
  });

  it('publishes a notification when onMutedSegmentsDetected is called', () => {
    const { wrapper } = setup();

    act(() => {
      wrapper.find(VodRestriction).prop('onMutedSegmentsDetected')!();
    });

    expect(mockPublish).toHaveBeenCalledTimes(1);
  });
});
