import type { FC, MutableRefObject, ReactNode } from 'react';
import { Pageview } from 'tachyon-event-tracker';
import { LatencyTransitionComplete } from 'tachyon-latency-tracker';
import type {
  TachyonPageInitialProps,
  TachyonPageProps,
  TachyonPageType,
} from 'tachyon-next-types';
import type { LoggedInCurrentUser } from 'tachyon-page-utils';
import { isValidQueryResponse } from 'tachyon-relay';
import { NodeNav, VerticalNav } from 'tachyon-tv-nav';
import type { StarshotPageExtensions } from '../../../../config';
import type { TopNavDisplayOptions } from '../../../common';
import { FullScreenLoadingContent, TopNav } from '../../../common';
import type { PageNavigationBehavior } from '../../types';

type PageRendererProps = {
  Page: TachyonPageType<
    TachyonPageInitialProps,
    TachyonPageProps,
    StarshotPageExtensions
  >;
  pageProps: TachyonPageProps;
  relayQueryResponse: unknown;
  user: MutableRefObject<LoggedInCurrentUser | null>;
};

function topNavOptions({
  displayBackButton,
  displayNavMenu,
}: PageNavigationBehavior): TopNavDisplayOptions | null {
  if (displayBackButton || displayNavMenu) {
    return {
      displayBackButton,
      displayNavMenu,
    } as TopNavDisplayOptions;
  }

  return null;
}

export const PageRenderer: FC<PageRendererProps> = ({
  Page,
  pageProps,
  relayQueryResponse,
  user,
}) => {
  const navigationBehavior = Page.navigationBehavior();
  const navOptions = topNavOptions(navigationBehavior);
  const { noFocusableContent, topNavBackground } = navigationBehavior;

  let elementCount: number;
  if (navOptions && !noFocusableContent) {
    elementCount = 2;
  } else if (navOptions || !noFocusableContent) {
    elementCount = 1;
  } else {
    elementCount = 0;
  }

  let pageContent: ReactNode;
  if (isValidQueryResponse(relayQueryResponse)) {
    user.current = Page.currentUser(relayQueryResponse);

    const componentProps = {
      ...pageProps,
      ...relayQueryResponse,
    };

    const innerContent = (
      <>
        <LatencyTransitionComplete />
        <Pageview {...Page.pageviewTracking(componentProps)} />
        <Page {...componentProps} />
      </>
    );

    pageContent = noFocusableContent ? (
      innerContent
    ) : (
      <NodeNav focusIndex={navOptions ? 1 : 0}>{innerContent}</NodeNav>
    );
  } else {
    pageContent = <FullScreenLoadingContent />;
  }

  return elementCount > 0 ? (
    <VerticalNav
      children={
        <>
          {navOptions && (
            <TopNav
              focusIndex={0}
              withBackground={topNavBackground}
              {...navOptions}
            />
          )}
          {pageContent}
        </>
      }
      elementCount={elementCount}
      focusIndex={0}
    />
  ) : (
    <>{pageContent}</>
  );
};

PageRenderer.displayName = 'PageRenderer';
