import type { CurrentUserResponse } from 'tachyon-page-utils';
import { defaultPageviewTracking, getCurrentUser } from 'tachyon-page-utils';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { VerticalNav } from 'tachyon-tv-nav';
import { FullScreenLoadingContent, TopNav } from '../../../common';
import type { PageNavigationBehavior, StarshotPage } from '../../types';
import { PageRenderer } from '.';

let displayNavMenu = true;
let topNavBackground = false;
let noFocusableContent = false;

const defaultNavigationBehavior = (): PageNavigationBehavior => ({
  displayNavMenu: !displayNavMenu ? undefined : true,
  noFocusableContent: !noFocusableContent ? undefined : true,
  topNavBackground: !topNavBackground ? undefined : true,
});

const TestPage: StarshotPage<{}, { foo: string }> = ({ foo }) => (
  <div>{foo}</div>
);
TestPage.pageviewTracking = defaultPageviewTracking;
TestPage.displayName = 'TestPage';
TestPage.navigationBehavior = defaultNavigationBehavior;
TestPage.currentUser = getCurrentUser;

describe(PageRenderer, () => {
  const setup = createShallowWrapperFactory(PageRenderer, () => ({
    Page: TestPage as StarshotPage,
    pageProps: { foo: 'bar' },
    relayQueryResponse: {},
    user: { current: null },
  }));

  beforeEach(() => {
    TestPage.navigationBehavior = defaultNavigationBehavior;
    displayNavMenu = true;
    topNavBackground = false;
  });

  it('renders the Top Nav and adjusts elementCount when the page opts in', () => {
    const { wrapper } = setup();
    expect(wrapper.find(VerticalNav).first()).toHaveProp({ elementCount: 2 });
    expect(wrapper.find(TopNav)).toHaveProp({ withBackground: undefined });
  });

  it('renders the Top Nav when the back button is desired but nav menu is not', () => {
    TestPage.navigationBehavior = (): PageNavigationBehavior => ({
      displayBackButton: true,
      noFocusableContent: !noFocusableContent ? undefined : true,
      topNavBackground: !topNavBackground ? undefined : true,
    });

    const { wrapper } = setup();
    expect(wrapper.find(TopNav)).toExist();
  });

  it('renders the Top Nav with a background when specified', () => {
    topNavBackground = true;
    const { wrapper } = setup();
    expect(wrapper.find(TopNav)).toHaveProp({ withBackground: true });
  });

  it('does not render the Top Nav and adjusts elementCount when the page opts out', () => {
    displayNavMenu = false;

    const { wrapper } = setup();
    expect(wrapper.find(VerticalNav).first()).toHaveProp({ elementCount: 1 });
    expect(wrapper.find(TopNav)).not.toExist();
  });

  it('renders the page content with page props when the query response is loaded', () => {
    const { wrapper } = setup();
    expect(wrapper.find(TestPage)).toHaveProp({ foo: 'bar' });
    expect(wrapper.find(FullScreenLoadingContent)).not.toExist();
  });

  it('renders the loading experience when the query response is null', () => {
    const { wrapper } = setup({ relayQueryResponse: null });
    expect(wrapper.find(TestPage)).not.toExist();
    expect(wrapper.find(FullScreenLoadingContent)).toExist();
  });

  it('sets current user if the query response is valid and contains a user', () => {
    const currentUserResponse: CurrentUserResponse = {
      currentUser: { id: '1234', login: 'therealderekt' },
    };

    const { props, wrapper } = setup({
      relayQueryResponse: currentUserResponse,
    });
    expect(props.user.current).toEqual(currentUserResponse.currentUser);

    wrapper.setProps({ relayQueryResponse: {} });
    expect(props.user.current).toBeNull();
  });

  it('forwards the relayQueryResponse to the Page', () => {
    const { wrapper } = setup({ relayQueryResponse: { derek: 'rocks' } });
    expect(wrapper.find(TestPage)).toHaveProp({ derek: 'rocks', foo: 'bar' });
  });

  it('does not render a root vertical nav if neither the top nav or focusable content exists', () => {
    displayNavMenu = false;
    noFocusableContent = true;

    const { wrapper } = setup();
    expect(wrapper.find(VerticalNav)).not.toExist();
    expect(wrapper.find(TestPage)).toExist();
  });
});
