import type { FC } from 'react';
import { useEffect } from 'react';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { NavigationRoot } from 'tachyon-tv-nav';
import { useUserIntent } from 'tachyon-user-intent';
import { KeyValue, legacyKeyboardEventHandler } from 'tachyon-utils';
import {
  KEY_NAV_THROTTLE_WAIT_MS,
  SCROLL_NAV_THROTTLE_WAIT_MS,
} from '../../../../../config';
import { routeNameFromPathname } from '../../../../../routing';
import { usePlaystation4Intent } from './usePlaystation4Intent';
import { useSwitchIntent } from './useSwitchIntent';
import { useWheelIntent } from './useWheelIntent';

// istanbul ignore next: trivial
export const StarshotNavigationRoot: FC = ({ children }) => {
  const { currentPathname } = useRouterUtils();
  const currentRouteName = routeNameFromPathname(currentPathname);
  const { triggerIntent } = useUserIntent();
  usePlaystation4Intent();
  useSwitchIntent();
  useWheelIntent();

  useEffect(() => {
    const navButtonListener = legacyKeyboardEventHandler((event) => {
      const key = event.key as KeyValue;
      switch (key) {
        case KeyValue.Down:
          triggerIntent('down');
          break;
        case KeyValue.Up:
          triggerIntent('up');
          break;
        case KeyValue.Right:
          triggerIntent('right');
          break;
        case KeyValue.Left:
          triggerIntent('left');
          break;
        default:
          return;
      }

      // Manufacturers such as Vestel build a psuedo tab-index system into their browser that
      // will respond to d-pad navigation. Disable input events from bubbling to those systems
      // in order to retain sole control of DOM focus management.
      event.preventDefault();
    });
    document.addEventListener('keydown', navButtonListener);

    return () => {
      document.removeEventListener('keydown', navButtonListener);
    };
  });

  return (
    <NavigationRoot
      children={children}
      focusSeed={currentRouteName}
      keyNavThrottleMs={KEY_NAV_THROTTLE_WAIT_MS}
      scrollNavThrottleMs={SCROLL_NAV_THROTTLE_WAIT_MS}
    />
  );
};

StarshotNavigationRoot.displayName = 'StarshotNavigationRoot';
