import type { FC } from 'react';
import { useMemo } from 'react';
import { RelayEnvironmentProvider } from 'react-relay/hooks';
import { useCurrentUser } from 'tachyon-auth';
import { useCustomTracking } from 'tachyon-event-tracker';
import type {
  FetchQueryOpts,
  QueryRendererProps,
  RecordMap,
  RelayEvent,
} from 'tachyon-relay';
import {
  RequestInfoRoot,
  TachyonQueryRenderer,
  initEnvironment,
} from 'tachyon-relay';
import { isBrowser } from 'tachyon-utils';

export type StarshotQueryRendererRootProps = Omit<
  QueryRendererProps,
  'environment'
> & {
  fetchQueryOpts: FetchQueryOpts;
  records: RecordMap | undefined;
};

export const StarshotQueryRendererRoot: FC<StarshotQueryRendererRootProps> = ({
  fetchQueryOpts,
  records,
  ...props
}) => {
  const { getAuthToken, logout } = useCurrentUser();
  const trackEvent = useCustomTracking<RelayEvent>();

  const optsMemo = useMemo(
    () =>
      isBrowser()
        ? {
            fetchQueryOpts: {
              ...fetchQueryOpts,
              authorization: {
                token: getAuthToken,
                unauthorizedHandler: logout,
              },
              onEvent: trackEvent,
            },
            records,
          }
        : { fetchQueryOpts, records },
    [logout, getAuthToken, fetchQueryOpts, records, trackEvent],
  );

  const environment = useMemo(() => initEnvironment(optsMemo), [optsMemo]);

  return (
    <RelayEnvironmentProvider environment={environment}>
      <RequestInfoRoot>
        <TachyonQueryRenderer {...props} environment={environment} />
      </RequestInfoRoot>
    </RelayEnvironmentProvider>
  );
};

StarshotQueryRendererRoot.displayName = 'StarshotQueryRendererRoot';
