import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import { fetchQuery } from 'react-relay/hooks';
import { getCurrentUserOnServer, logoutOnServer } from 'tachyon-auth-server';
import type { LogPayload } from 'tachyon-logger';
import { logger } from 'tachyon-logger';
import type { TachyonPageContext, TachyonPageType } from 'tachyon-next-types';
import type { FetchQueryOpts, RecordMap } from 'tachyon-relay';
import { initEnvironment, isValidQueryResponse } from 'tachyon-relay';
import { HTTPStatusCode } from 'tachyon-type-library';
import { buildCacheHeader } from 'tachyon-utils';
import type { SetRequired } from 'type-fest';

type ServersideTachyonPageContext<
  Ctx extends TachyonPageContext = TachyonPageContext,
> = Ctx & SetRequired<Ctx, 'req' | 'res'>;

export function isServersideContext<Ctx extends TachyonPageContext>(
  ctx: Ctx,
): ctx is ServersideTachyonPageContext<Ctx> {
  return ctx.req !== undefined && ctx.res !== undefined;
}

type GetServersideInitialPropsOpts<
  Ctx extends ServersideTachyonPageContext,
  Page extends TachyonPageType,
> = {
  Component: Page;
  fetchQueryOpts: FetchQueryOpts;
  relayQueryVariables: RelayQueryVariables;
  tachyonCtx: Ctx;
};

type ServersideInitialPropsError = {
  isGQLPartialSuccess: false;
  isLoggedInServerside: boolean;
  relayQueryRecords: undefined;
  serversideError: true;
};

type ServersideInitialPropsSuccess = {
  isGQLPartialSuccess: boolean;
  isLoggedInServerside: boolean;
  relayQueryRecords: RecordMap | undefined;
  serversideError: false;
};

export type ServersideInitialProps =
  | ServersideInitialPropsError
  | ServersideInitialPropsSuccess;

export async function getServersideInitialProps<
  Ctx extends ServersideTachyonPageContext,
  Page extends TachyonPageType,
>({
  Component,
  fetchQueryOpts,
  relayQueryVariables,
  tachyonCtx,
}: GetServersideInitialPropsOpts<Ctx, Page>): Promise<ServersideInitialProps> {
  tachyonCtx.res.setHeader(
    'Cache-Control',
    buildCacheHeader({ totalCacheLife: 0 }),
  );

  const currentUser = getCurrentUserOnServer(tachyonCtx.req, tachyonCtx.res);
  let isLoggedInServerside = currentUser.isLoggedIn;
  let isGQLPartialSuccess = false;

  let relayQueryRecords;
  if (Component.query) {
    const relayEnvironment = initEnvironment({
      fetchQueryOpts: {
        ...fetchQueryOpts,
        authorization: {
          token: currentUser.authorizationToken,
          unauthorizedHandler: () => {
            isLoggedInServerside = false;
            logoutOnServer(tachyonCtx.res);
          },
        },
        nonFatalErrorsHandler: () => {
          isGQLPartialSuccess = true;
        },
      },
    });

    let relayQueryResponse;
    try {
      relayQueryResponse = await fetchQuery(
        relayEnvironment,
        Component.query,
        relayQueryVariables,
      ).toPromise();
    } catch (error) {
      logger.info(error as LogPayload);
      // CSI uses this status code to uniquely identify errors from mobile
      // web that originated from the GQL API Gateway or a backend service.
      tachyonCtx.res.statusCode = HTTPStatusCode.BadGateway;
    }

    if (isValidQueryResponse(relayQueryResponse)) {
      // each page is responsible for actually rendering its own 404, this
      // only controls status setting
      const componentInitialProps =
        (await Component.getInitialProps?.(tachyonCtx)) ?? {};

      const componentProps = {
        ...componentInitialProps,
        ...relayQueryResponse,
      };

      const isNotFoundServerside =
        Component.isNotFoundServerside?.(componentProps);

      if (isNotFoundServerside) {
        tachyonCtx.res.statusCode = HTTPStatusCode.NotFound;
      }

      relayQueryRecords = relayEnvironment.getStore().getSource().toJSON();
    }
  }

  const statusCode = tachyonCtx.res.statusCode;
  const error500 = 500 <= statusCode && statusCode <= 599;
  const serversideError = tachyonCtx.err ?? error500;

  // Set statusCode to 5XX so we don't cache or index partial success pages.
  // This does not set serversideError.
  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  if (isGQLPartialSuccess) {
    tachyonCtx.res.statusCode = HTTPStatusCode.BadGateway;
  }

  return serversideError
    ? {
        isGQLPartialSuccess: false,
        isLoggedInServerside,
        relayQueryRecords: undefined,
        serversideError: true,
      }
    : {
        isGQLPartialSuccess,
        isLoggedInServerside,
        relayQueryRecords,
        serversideError: false,
      };
}
