import App from 'next/app';
import Head from 'next/head';
import type { FC } from 'react';
import { useCallback, useRef } from 'react';
import type { Variables as RelayQueryVariables } from 'react-relay/hooks';
import type { DynamicSettings } from 'tachyon-dynamic-settings';
import { DynamicSettingsRoot } from 'tachyon-dynamic-settings';
import type { AppEnvironment } from 'tachyon-environment';
import { GamepadInputRoot } from 'tachyon-gamepad-input';
import { TachyonIntlRoot } from 'tachyon-intl';
import type { ClientSideErrorBoundaryProps } from 'tachyon-more-ui';
import { ClientSideErrorBoundary, CoreUiSsrRoot } from 'tachyon-more-ui';
import { RouterUtilsRoot } from 'tachyon-next-routing-utils';
import type {
  TachyonAppBaseInitialProps,
  TachyonAppBaseProps,
  TachyonAppFC,
  TachyonPageInitialProps,
} from 'tachyon-next-types';
import { NotificationContextRoot } from 'tachyon-notification';
import type { LoggedInCurrentUser } from 'tachyon-page-utils';
import { useErrorPath } from 'tachyon-page-utils';
import type { FetchQueryOpts, QueryRendererRenderProps } from 'tachyon-relay';
import { UserIntentRoot } from 'tachyon-user-intent';
import { isBrowser } from 'tachyon-utils';
import { Background } from 'twitch-core-ui';
import type {
  StarshotDynamicSettings,
  StarshotPageContextExtensions,
  StarshotPageExtensions,
  StarshotRequestExtensions,
} from '../../../config';
import {
  EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY,
  PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY,
  REDIRECT_ROUTE_PARAM_KEY,
  SEARCH_TERM_QUERY_PARAM,
} from '../../../config';
import { AppVisibilityRoot, Toaster } from '../../common';
import { InternalServerError } from '../../errors';
import {
  DevMouseAsWandObserver,
  NativeAppProxyRoot,
  RecentUserInputRoot,
  StarshotVisualDebugModeRoot,
  WandActivityRoot,
} from '../../framework';
import { PageRenderer } from './PageRenderer';
import { configure } from './configure';
import {
  StarshotChatAllowedRoot,
  StarshotCurrentUserRoot,
  StarshotDebugRoot,
  StarshotDiscoveryTrackingRoot,
  StarshotEnvironmentRoot,
  StarshotEventReporterRoot,
  StarshotEventTrackerRoot,
  StarshotExperimentsRoot,
  StarshotLatencyTrackerRoot,
  StarshotNavigationRoot,
  StarshotQueryRendererRoot,
} from './context-root-adapters';
import type { ServersideInitialProps } from './getServersideInitialProps';
import {
  getServersideInitialProps,
  isServersideContext,
} from './getServersideInitialProps';

configure();

// list of query params to protect from being stripped by RouterUtilsRoot
// (allows them to persist through reloads)
const PRESERVED_QUERY_PARAMS = [
  // Allow QA to override the assigned bucket
  EXPERIMENT_OVERRIDES_QUERY_PARAM_KEY,
  // Index into a VOD at a specific time point
  PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY,
  // Used by login to redirect after sucessful login
  REDIRECT_ROUTE_PARAM_KEY,
  // Used for search
  SEARCH_TERM_QUERY_PARAM,
];

// non-localized error UI of last resort
const RootErrorMessage: FC = () => (
  <span>An error occurred. Please refresh the page.</span>
);
RootErrorMessage.displayName = 'RootErrorMessage';

/**
 * StarshotPersistedAppInitialProps is comprised of values that we need to transport
 * from the server to client, and we lean on next's `getInitialProps`
 * serialization functionality to achieve that. The values are set on the
 * server, consumed from __NEXT_DATA__ via normal next functionality for the
 * first page render, and then manually re-consumed on subsequent app re-renders
 * to ensure their continued availability.
 */
type StarshotPersistedAppInitialProps = Pick<
  StarshotRequestExtensions,
  'deviceId' | 'experimentBucket' | 'intlData' | 'platform'
> & {
  appEnvironment: AppEnvironment;
  dynamicSettings: DynamicSettings<StarshotDynamicSettings>;
  language: string;
};

// Be very conservative about what is put in here, since all of this will get
// serialized for being sent to the client and big/complex objects both don't
// do very well with this and also cause a ton of page bloat.
type StarshotAppInitialProps = Partial<ServersideInitialProps> &
  StarshotPersistedAppInitialProps &
  TachyonAppBaseInitialProps & {
    currentAsPath: string;
    fetchQueryOpts: FetchQueryOpts;
    relayQueryVariables: RelayQueryVariables;
  };

export type StarshotAppProps = StarshotAppInitialProps & TachyonAppBaseProps;

export const StarshotApp: TachyonAppFC<
  StarshotAppInitialProps,
  StarshotAppProps,
  StarshotRequestExtensions,
  StarshotPageContextExtensions,
  StarshotPageExtensions
> = ({
  Component,
  appEnvironment,
  currentAsPath,
  dynamicSettings,
  experimentBucket,
  fetchQueryOpts,
  intlData,
  isLoggedInServerside,
  language,
  pageProps,
  platform,
  relayQueryRecords,
  relayQueryVariables,
  serversideError,
}) => {
  const user = useRef<LoggedInCurrentUser | null>(null);
  const getUserTracking = useCallback(() => user.current ?? undefined, [user]);
  const [erroringPath, onError] = useErrorPath(currentAsPath);

  const commonErrorBoundaryProps: Pick<
    ClientSideErrorBoundaryProps,
    'app' | 'currentPath' | 'isDevelopment'
  > = {
    app: 'starshot',
    currentPath: currentAsPath,
    isDevelopment: process.env.NODE_ENV === 'development',
  };

  return (
    <ClientSideErrorBoundary
      {...commonErrorBoundaryProps}
      boundaryName="StarshotRootBoundary"
      errorStateComponent={RootErrorMessage}
      isRootBoundary
    >
      <Head>
        <meta
          content="width=device-width, height=device-height, initial-scale=1, user-scalable=no, viewport-fit=cover"
          name="viewport"
        />
      </Head>
      <DevMouseAsWandObserver />
      <RouterUtilsRoot preservedParams={PRESERVED_QUERY_PARAMS}>
        <DynamicSettingsRoot settings={dynamicSettings}>
          <StarshotEnvironmentRoot
            appEnvironment={appEnvironment}
            language={language}
            platform={platform}
          >
            <NativeAppProxyRoot>
              <StarshotEventReporterRoot>
                <StarshotDebugRoot />
                <TachyonIntlRoot data={intlData}>
                  <StarshotEventTrackerRoot
                    getUserTracking={getUserTracking}
                    inError={serversideError ?? !!erroringPath}
                  >
                    <StarshotLatencyTrackerRoot>
                      <StarshotExperimentsRoot
                        bucket={experimentBucket}
                        debug={appEnvironment !== 'production'}
                        experiments={dynamicSettings.experiments}
                      >
                        <StarshotCurrentUserRoot
                          isLoggedInServerside={isLoggedInServerside}
                        >
                          <StarshotChatAllowedRoot>
                            <UserIntentRoot>
                              <GamepadInputRoot>
                                <RecentUserInputRoot>
                                  <StarshotNavigationRoot>
                                    <AppVisibilityRoot>
                                      <StarshotDiscoveryTrackingRoot>
                                        <WandActivityRoot>
                                          <NotificationContextRoot>
                                            <StarshotVisualDebugModeRoot>
                                              <ClientSideErrorBoundary
                                                {...commonErrorBoundaryProps}
                                                boundaryName="StarshotAppBoundary"
                                                errorStateComponent={
                                                  InternalServerError
                                                }
                                                onError={onError}
                                              >
                                                <CoreUiSsrRoot
                                                  appRootElementId="__next"
                                                  background={
                                                    Background.Inherit
                                                  }
                                                  disableHoverCSS
                                                  theme="dark"
                                                >
                                                  {serversideError ? (
                                                    <InternalServerError />
                                                  ) : (
                                                    <StarshotQueryRendererRoot
                                                      fetchPolicy="store-and-network"
                                                      fetchQueryOpts={
                                                        fetchQueryOpts
                                                      }
                                                      query={Component.query}
                                                      records={
                                                        relayQueryRecords
                                                      }
                                                      render={({
                                                        error,
                                                        props:
                                                          relayQueryResponse,
                                                      }: QueryRendererRenderProps) => {
                                                        if (error) {
                                                          // throw here because we've already
                                                          // logged and this will take us to
                                                          // error page rendering
                                                          throw error;
                                                        }

                                                        return (
                                                          <PageRenderer
                                                            Page={Component}
                                                            pageProps={
                                                              pageProps
                                                            }
                                                            relayQueryResponse={
                                                              relayQueryResponse
                                                            }
                                                            user={user}
                                                          />
                                                        );
                                                      }}
                                                      variables={
                                                        relayQueryVariables
                                                      }
                                                    />
                                                  )}
                                                  <Toaster />
                                                </CoreUiSsrRoot>
                                              </ClientSideErrorBoundary>
                                            </StarshotVisualDebugModeRoot>
                                          </NotificationContextRoot>
                                        </WandActivityRoot>
                                      </StarshotDiscoveryTrackingRoot>
                                    </AppVisibilityRoot>
                                  </StarshotNavigationRoot>
                                </RecentUserInputRoot>
                              </GamepadInputRoot>
                            </UserIntentRoot>
                          </StarshotChatAllowedRoot>
                        </StarshotCurrentUserRoot>
                      </StarshotExperimentsRoot>
                    </StarshotLatencyTrackerRoot>
                  </StarshotEventTrackerRoot>
                </TachyonIntlRoot>
              </StarshotEventReporterRoot>
            </NativeAppProxyRoot>
          </StarshotEnvironmentRoot>
        </DynamicSettingsRoot>
      </RouterUtilsRoot>
    </ClientSideErrorBoundary>
  );
};

StarshotApp.getInitialProps = async (ctx) => {
  const { Component, ctx: tachyonCtx } = ctx;

  const persistedProps: StarshotPersistedAppInitialProps = isBrowser()
    ? {
        /* eslint-disable @typescript-eslint/no-unsafe-assignment */
        appEnvironment: window.__NEXT_DATA__.props.appEnvironment,
        deviceId: window.__NEXT_DATA__.props.deviceId,
        dynamicSettings: window.__NEXT_DATA__.props.dynamicSettings,
        experimentBucket: window.__NEXT_DATA__.props.experimentBucket,
        intlData: window.__NEXT_DATA__.props.intlData,
        language: window.__NEXT_DATA__.props.language,
        platform: window.__NEXT_DATA__.props.platform,
        /* eslint-enable @typescript-eslint/no-unsafe-assignment */
      }
    : {
        /* eslint-disable @typescript-eslint/no-non-null-assertion */
        appEnvironment: tachyonCtx.req!.tachyon.appEnvironment,
        deviceId: tachyonCtx.req!.tachyon.deviceId,
        dynamicSettings: tachyonCtx.req!.tachyon.dynamicSettings,
        experimentBucket: tachyonCtx.req!.tachyon.experimentBucket,
        intlData: tachyonCtx.req!.tachyon.intlData,
        language: tachyonCtx.req!.tachyon.intlData.preferredLanguageTags[0],
        platform: tachyonCtx.req!.tachyon.platform,
        /* eslint-enable @typescript-eslint/no-non-null-assertion */
      };

  // update context before calling App.getInitialProps to make values
  // available to current page component
  tachyonCtx.language = persistedProps.language;
  tachyonCtx.platform = persistedProps.platform;
  const initialProps = await App.getInitialProps(ctx);

  const fetchQueryOpts: FetchQueryOpts = {
    deviceId: persistedProps.deviceId,
    language: persistedProps.language,
  };

  const relayQueryVariables: RelayQueryVariables =
    (initialProps.pageProps as TachyonPageInitialProps).queryVariables ?? {};

  const serverSideInitialProps =
    // typeof window and tachyonCtx.req/res should move in sync, here just for types
    typeof window === 'undefined' && isServersideContext(tachyonCtx)
      ? await getServersideInitialProps({
          Component,
          fetchQueryOpts,
          relayQueryVariables,
          tachyonCtx,
        })
      : undefined;

  return {
    ...initialProps,
    ...persistedProps,
    ...serverSideInitialProps,
    currentAsPath: tachyonCtx.asPath!, // eslint-disable-line @typescript-eslint/no-non-null-assertion
    fetchQueryOpts,
    relayQueryVariables,
  };
};

StarshotApp.displayName = 'StarshotApp';
