import { lorem } from 'faker';
import Head from 'next/head';
import { Platform } from 'tachyon-environment';
import { mockIntlData } from 'tachyon-intl';
import type {
  TachyonPage,
  TachyonPageInitialProps,
  TachyonPageProps,
} from 'tachyon-next-types';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { StarshotPageExtensions } from '../../../config';
import { InternalServerError } from '../../errors';
import type { StarshotQueryRendererRootProps } from './context-root-adapters';
import { StarshotQueryRendererRoot } from './context-root-adapters';
import { getServersideInitialProps } from './getServersideInitialProps';
import type { StarshotAppProps } from '.';
import { StarshotApp } from '.';

/* supress console messages when running tests */
jest.mock('tachyon-logger', () => ({ logger: { debug: jest.fn() } }));
const mockConsoleError = jest.spyOn(console, 'error').mockImplementation();
afterAll(() => {
  mockConsoleError.mockRestore();
});

const mockStarshotQueryRenderer = StarshotQueryRendererRoot as jest.Mock;
jest.mock('./context-root-adapters', () => ({
  ...jest.requireActual('./context-root-adapters'),
  StarshotQueryRendererRoot: jest.fn((props) =>
    props.render({ error: false, props: true }),
  ),
}));

jest.mock('tachyon-next-routing-utils', () => {
  return {
    ...jest.requireActual('tachyon-next-routing-utils'),
    RouterUtilsRoot: (props: any) => props.children,
  };
});

jest.mock('./configure', () => ({
  configure: jest.fn(),
}));

const mockHead = Head as unknown as jest.Mock;
jest.mock('next/head', () => ({
  __esModule: true,
  default: jest.fn(() => <div />),
}));

jest.mock('next/app', () => {
  const app = () => false;
  app.getInitialProps = () => ({ pageProps: { queryVariables: {} } });
  return {
    __esModule: true,
    default: app,
  };
});

jest.mock('tachyon-debug-reporter', () => ({
  configTachyonDebug: jest.fn(),
}));

const mockGetServersideInitialProps = getServersideInitialProps as jest.Mock;
jest.mock('./getServersideInitialProps', () => ({
  ...jest.requireActual('./getServersideInitialProps'),
  getServersideInitialProps: jest.fn(() => ({ isLoggedInServerside: false })),
}));

jest.mock('../../errors', () => ({
  InternalServerError: () => <div />,
}));

const mockPage = (
  renderer: any = () => <div />,
): TachyonPage<
  TachyonPageInitialProps,
  TachyonPageProps,
  StarshotPageExtensions
> => {
  const page = renderer;
  page.pageviewTracking = defaultPageviewTracking;
  page.currentUser = () => null;
  page.navigationBehavior = () => ({ displayNavMenu: true });

  return page;
};

describe(StarshotApp, () => {
  const pageProps = { foo: 'bar' };
  const intlData = mockIntlData();

  const setup = createMountWrapperFactory(
    StarshotApp as any,
    (): StarshotAppProps => ({
      Component: mockPage(),
      appEnvironment: 'development',
      currentAsPath: '/',
      deviceId: 'deviceId',
      dynamicSettings: {
        environment: 'development',
        experiments: {},
        spadeUrl: '//spade',
      },
      experimentBucket: 'a',
      fetchQueryOpts: {},
      intlData,
      isLoggedInServerside: false,
      language: intlData.preferredLanguageTags[0],
      pageProps: pageProps as any,
      platform: Platform.StarshotWeb,
      relayQueryRecords: {},
      relayQueryVariables: {},
      router: {} as any,
      serversideError: false,
    }),
  );

  it('renders <InternalServerError /> when there is an SSR error', () => {
    const { wrapper } = setup({ serversideError: true });

    expect(wrapper.find(InternalServerError)).toExist();
  });

  it('renders the StarshotAppBoundary when the <Page/> fails', () => {
    const { wrapper } = setup({
      Component: mockPage(() => {
        throw new Error('oh no');
      }),
    });

    expect(wrapper.find({ boundaryName: 'StarshotAppBoundary' })).toExist();
    expect(wrapper.find(InternalServerError)).toExist();
  });

  it('renders the StarshotRootBoundary when the app fails', () => {
    mockHead.mockImplementationOnce(() => {
      throw new Error('oh no');
    });
    const { wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'StarshotRootBoundary' })).toExist();
    expect(wrapper).toIncludeText(
      'An error occurred. Please refresh the page.',
    );
  });

  it('renders the <Page /> when there is not an SSR error', () => {
    const { props, wrapper } = setup();

    expect(wrapper.find(props.Component as any)).toExist();
  });

  it('renders the StarshotAppBoundary when the QueryRenderer has an error', () => {
    mockStarshotQueryRenderer.mockImplementationOnce(
      (props: StarshotQueryRendererRootProps) =>
        props.render({
          error: new Error('oh no'),
          props: true,
          retry: jest.fn(),
        }),
    );
    const { wrapper } = setup();

    expect(wrapper.find({ boundaryName: 'StarshotAppBoundary' })).toExist();
    expect(wrapper.find(InternalServerError)).toExist();
  });
});

describe('StarshotApp.getInitialProps', () => {
  describe('client side', () => {
    const intlData = mockIntlData();

    const mockNextData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
      experimentBucket: lorem.word(),
      intlData,
      language: intlData.preferredLanguageTags[0],
      platform: Platform.StarshotWeb,
    };

    beforeEach(() => {
      global.window.__NEXT_DATA__ = { props: mockNextData } as any;
    });

    // @ts-expect-error: tests
    afterEach(() => delete global.window.__NEXT_DATA__);

    it('extracts __NEXT_DATA__ fields', async () => {
      const ip = await StarshotApp.getInitialProps({ ctx: {} } as any);

      expect(ip).toEqual(expect.objectContaining(mockNextData));
    });
  });

  describe('server side', () => {
    let mockWindow: any;
    beforeEach(() => {
      mockWindow = jest.spyOn(global as any, 'window', 'get');
      mockWindow.mockImplementation(() => undefined);
    });
    afterEach(() => {
      mockWindow.mockRestore();
    });

    const mockReqData = {
      appEnvironment: 'staging',
      dynamicSettings: { spadeUrl: lorem.word() },
      intlData: mockIntlData(),
      platform: Platform.StarshotWeb,
    };

    const experimentBucket = 'b';
    const mockCtx = {
      ctx: { req: { tachyon: { ...mockReqData, experimentBucket } }, res: {} },
    };

    it('extracts req fields when', async () => {
      const ip = await StarshotApp.getInitialProps(mockCtx as any);

      expect(ip).toEqual(
        expect.objectContaining({
          ...mockReqData,
          experimentBucket,
          language: mockReqData.intlData.preferredLanguageTags[0],
        }),
      );
    });

    it('invokes getServersideInitialProps', async () => {
      await StarshotApp.getInitialProps(mockCtx as any);

      expect(mockGetServersideInitialProps).toHaveBeenCalledTimes(1);
    });
  });
});
