import type { DocumentInitialProps, DocumentProps } from 'next/document';
import Document, { Head, Html, Main, NextScript } from 'next/document';
import { PulsarCorePreloadASMJS } from 'pulsar';
import { ServerStyleSheet } from 'styled-components';
import { PerformancePolyfill } from 'tachyon-latency-tracker';
import { Fonts, TvStyles } from 'tachyon-more-ui';
import type { TachyonDocumentContext } from 'tachyon-next-types';
import { PolyfillPreload, PolyfillScript } from 'tachyon-polyfill';
import { GraphqlPreconnect } from 'tachyon-relay';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import type { StarshotRequestExtensions } from '../../../config';
import { SwitchHead } from './SwitchHead';

export type StarshotDocumentInitialProps = DocumentInitialProps & {
  language: string;
};

export type StarshotDocumentProps = DocumentProps &
  StarshotDocumentInitialProps;

// istanbul ignore next: hard to test, low value
/**
 * Root-level React component that is only rendered server-side, see:
 * https://nextjs.org/docs/#custom-document
 */
export class StarshotDocument extends Document<StarshotDocumentProps> {
  public static async getInitialProps(
    context: TachyonDocumentContext<StarshotRequestExtensions>,
  ): Promise<StarshotDocumentInitialProps> {
    const sheet = new ServerStyleSheet();
    const originalRenderPage = context.renderPage;

    try {
      context.renderPage = () =>
        originalRenderPage({
          enhanceApp: (App) => (props) =>
            sheet.collectStyles(<App {...props} />),
        });

      const initialProps = await super.getInitialProps(context);
      return {
        ...initialProps,
        language: context.req!.tachyon.intlData.preferredLanguageTags[0], // eslint-disable-line @typescript-eslint/no-non-null-assertion
        styles: sheet.getStyleElement(),
      };
    } finally {
      sheet.seal();
    }
  }

  public render(): JSX.Element {
    const { language } = this.props;

    // The "style" prop is set directly on html to prevent a flash-of-white on initial
    // page load since html is parsed before anything in head including the
    // flushed CSS used for styled components that are SSR'd.
    /* eslint-disable react/forbid-dom-props */
    return (
      <Html
        lang={language}
        style={{
          background: DarkThemeMap['color-background-body'],
          // When we hide the Switch's footer chrome, the Switch browser does
          // not properly cause a re-layout, meaning that the viewport height
          // and root containing block are not updated and we wind up with
          // weird letterboxing of the app. Using calc generates a static value
          // that is not dependent on the broken viewport height.
          minHeight: 'calc(100vw * 16 / 9)',
        }}
      >
        <Head>
          <SwitchHead />
          <PerformancePolyfill />
          <link href="https://static-cdn.jtvnw.net" rel="preconnect" />
          <PulsarCorePreloadASMJS />
          <GraphqlPreconnect />
          <PolyfillPreload locale={language} />
          <Fonts />
          <TvStyles />
        </Head>
        <body style={{ background: DarkThemeMap['color-background-body'] }}>
          <Main />
          <PolyfillScript locale={language} />
          <NextScript />
        </body>
      </Html>
    );
  }
}
