import { PlayerType, createClipManifestUrl, useClipAnalytics } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { useEffectOnce } from 'tachyon-utils';
import { getPulsarClipBackend } from '../../../config';
import { PlayerLoadingOverlay } from '../PlayerLoadingOverlay';
import type { PulsarBackendLoaderProps } from '../PulsarBackendLoader';
import { PulsarBackendLoader } from '../PulsarBackendLoader';
import { useDiscoveryTrackingEvent } from '../useDiscoveryTrackingEvent';
import { useRefetchPlaybackAccessToken } from '../useRefetchPlaybackAccessToken';
import type { ClipPlayer_clip$key } from './__generated__/ClipPlayer_clip.graphql';
import type { ClipPlayer_currentUser$key } from './__generated__/ClipPlayer_currentUser.graphql';
import type { ClipPlayer_token$key } from './__generated__/ClipPlayer_token.graphql';

export type {
  ClipPlayer_token$key,
  ClipPlayer_token,
} from './__generated__/ClipPlayer_token.graphql';

const clipPlayerClipFragment = graphql`
  fragment ClipPlayer_clip on Clip {
    id
    game {
      name
    }
    broadcaster {
      id
      login
      roles {
        isPartner
      }
      self {
        subscriptionBenefit {
          id
        }
      }
    }
    slug
  }
`;

const clipManifestFragment = graphql`
  fragment ClipPlayer_token on Query
  @refetchable(queryName: "ClipPlayer_AccessToken") {
    clip(slug: $slug) @skip(if: $skipPlayToken) {
      playbackAccessToken(
        params: { platform: $platform, playerType: $playerType }
      ) {
        signature
        value
      }
      videoQualities {
        frameRate
        quality
        sourceURL
      }
    }
  }
`;

const clipTokenRefetchQuery = graphql`
  query ClipPlayer_Query(
    $slug: ID!
    $playerType: String!
    $platform: String!
    $skipPlayToken: Boolean!
  ) {
    ...ClipPlayer_token
  }
`;

const clipPlayerCurrentUserFragment = graphql`
  fragment ClipPlayer_currentUser on User {
    id
    hasTurbo
  }
`;

export type ClipPlayerProps = Omit<
  PulsarBackendLoaderProps,
  'backend' | 'src'
> & {
  clip: ClipPlayer_clip$key;
  clipToken: ClipPlayer_token$key;
  currentUser: ClipPlayer_currentUser$key | null;
  onManifestUrlCreated?: (url: string | null) => void;
};

export const ClipPlayer: FC<ClipPlayerProps> = ({
  clip: clipRef,
  clipToken: clipTokenRef,
  currentUser: currentUserRef,
  onManifestUrlCreated,
  ...playerProps
}) => {
  const {
    common: { platform },
  } = useStaticEnvironment();
  const currentUser = useFragment(
    clipPlayerCurrentUserFragment,
    currentUserRef,
  );
  const clip = useFragment(clipPlayerClipFragment, clipRef);

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const playerType = PlayerType.Pulsar;

  const onTrackingEvent = useDiscoveryTrackingEvent();
  useClipAnalytics({
    clip,
    currentUser,
    onTrackingEvent,
    tracking: {
      playerType,
      sessionId: sessionID,
    },
  });

  const { refetchToken, token: clipToken } = useRefetchPlaybackAccessToken({
    playerType,
    refetchQuery: clipTokenRefetchQuery,
    slug: clip.slug,
    tokenFragment: clipManifestFragment,
    tokenRef: clipTokenRef,
  });

  useEffectOnce(() => {
    if (!clipToken.clip?.playbackAccessToken) {
      refetchToken();
    }
  });

  const manifestUrl = createClipManifestUrl(clipToken.clip, {
    maxQuality: platform === Platform.Switch ? 720 : 1080,
  });

  // Specifically for clips, we set the manifest url before checking for null
  // since null values indicate the clip is not playable
  onManifestUrlCreated?.(manifestUrl);

  return (
    <PlayerLoadingOverlay>
      <PulsarBackendLoader
        backend={getPulsarClipBackend(platform)}
        src={manifestUrl}
        {...playerProps}
      />
    </PlayerLoadingOverlay>
  );
};

ClipPlayer.displayName = 'ClipPlayer';
