import { datatype } from 'faker';
import {
  createClipManifestUrl,
  mockPlayerController,
  usePlayerController,
} from 'pulsar';
import { useFragment } from 'react-relay/hooks';
import {
  Platform,
  createMockStaticEnvironmentContext,
  getOtherEnvMock,
} from 'tachyon-environment';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { OmitRefType } from 'tachyon-type-library';
import { PulsarBackendLoader } from '../PulsarBackendLoader';
import { useRefetchPlaybackAccessToken } from '../useRefetchPlaybackAccessToken';
import type { ClipPlayer_clip } from './__generated__/ClipPlayer_clip.graphql';
import type { ClipPlayer_token } from './__generated__/ClipPlayer_token.graphql';
import { ClipPlayer } from '.';

jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useFragment: jest.fn(),
}));
const mockUseFragment = useFragment as jest.Mock;

jest.mock('../PulsarBackendLoader', () => ({
  PulsarBackendLoader: jest.fn(() => <div />),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerController: jest.fn(),
}));

jest.mock('../useRefetchPlaybackAccessToken', () => ({
  useRefetchPlaybackAccessToken: jest.fn(),
}));
const mockUseRefetchPlaybackAccessToken =
  useRefetchPlaybackAccessToken as jest.Mock;

function mockFragments(
  overrides: Partial<ClipPlayer_clip> = {},
): [OmitRefType<ClipPlayer_clip>, { token: OmitRefType<ClipPlayer_token> }] {
  return [
    {
      broadcaster: null,
      game: null,
      id: datatype.uuid(),
      slug: 'threalderektpanicbasket',
      ...overrides,
    },
    {
      token: {
        clip: {
          playbackAccessToken: {
            signature: '123',
            value: 'some-token',
          },
          videoQualities: [
            { frameRate: 60, quality: '720', sourceURL: 'https://some.mp4' },
          ],
        },
      },
    },
  ];
}

describe(ClipPlayer, () => {
  const setup = createMountWrapperFactory(ClipPlayer, () => ({
    clip: { ' $fragmentRefs': { ClipPlayer_clip: true } },
    clipToken: { ' $fragmentRefs': { ClipPlayer_token: true } },
    currentUser: null,
    onManifestUrlCreated: jest.fn(),
  }));

  const mockedPlayerController = mockPlayerController();

  beforeEach(() => {
    mockUseFragment.mockReset();
    mockUseRefetchPlaybackAccessToken.mockReset();
    // Mock the first call, used for `currentUser`
    mockUseFragment.mockImplementationOnce(() => null);
    (usePlayerController as jest.Mock).mockImplementation(
      () => mockedPlayerController,
    );
  });

  describe('Nintendo Switch', () => {
    it('enables ad resolution matching for the manifest URL', () => {
      const [channel, tokenResponse] = mockFragments();
      mockUseFragment.mockImplementationOnce(() => channel);
      mockUseRefetchPlaybackAccessToken.mockImplementationOnce(
        () => tokenResponse,
      );

      const { props, wrapper } = setup(
        {},
        {
          wrappingContexts: [
            createMockStaticEnvironmentContext(
              getOtherEnvMock({ platform: Platform.Switch }),
            ),
          ],
        },
      );

      expect(props.onManifestUrlCreated).toHaveBeenCalledTimes(1);
      expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
        src: createClipManifestUrl(tokenResponse.token.clip, {
          maxQuality: 720,
        }),
      });
    });
  });

  it('renders a PulsarBackendLoader with the expected params', () => {
    const [channel, tokenResponse] = mockFragments();
    mockUseFragment.mockImplementationOnce(() => channel);
    mockUseRefetchPlaybackAccessToken.mockImplementationOnce(
      () => tokenResponse,
    );

    const { props, wrapper } = setup();

    expect(props.onManifestUrlCreated).toHaveBeenCalledTimes(1);
    expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
      src: createClipManifestUrl(tokenResponse.token.clip, {
        maxQuality: 1080,
      }),
    });
  });
});
