import { usePlaybackState } from 'pulsar';
import { act } from 'react-dom/test-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { useScrubControls } from '../ScrubControls';
import {
  HideablePlayPauseButton,
  ScPausedIndicator,
  ScPlayPauseButton,
} from '.';

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlaybackState: jest.fn(),
}));

jest.mock('../ScrubControls', () => ({
  ...jest.requireActual('../ScrubControls'),
  useScrubControls: jest.fn(),
}));

const mockPlaybackState = usePlaybackState as jest.Mock;
const mockUseScrubControls = useScrubControls as jest.Mock;

describe(HideablePlayPauseButton, () => {
  const setup = createMountWrapperFactory(HideablePlayPauseButton);
  const setupMocks = () => {
    mockPlaybackState.mockReturnValue('Initializing');
    mockUseScrubControls.mockImplementation(() => ({
      subscribeToScrubbingPositionSeconds: jest.fn(() => () => undefined),
    }));
  };

  beforeEach(() => {
    jest.resetAllMocks();
    setupMocks();
  });

  afterEach(() => {
    jest.restoreAllMocks();
  });

  it('hides when playerstate is not idle', () => {
    mockPlaybackState.mockReturnValueOnce('Playing');
    const { wrapper } = setup();

    expect(wrapper.find(ScPausedIndicator).props()['$hide']).toEqual(true);
  });

  it('hides when we are scrubbing', () => {
    const mockSubscribeToScrubbingPositionSeconds = jest.fn();

    mockUseScrubControls.mockImplementationOnce(() => ({
      subscribeToScrubbingPositionSeconds:
        mockSubscribeToScrubbingPositionSeconds,
      togglePlay: jest.fn(),
    }));

    const { wrapper } = setup();

    const componentScrubHandler =
      mockSubscribeToScrubbingPositionSeconds.mock.calls[0][0];

    act(() => {
      componentScrubHandler(200);
    });

    wrapper.update();

    expect(wrapper.find(ScPausedIndicator).props()['$hide']).toEqual(true);
  });

  it('displays when we are not scrubbing and the playstate is idle', () => {
    const mockSubscribeToScrubbingPositionSeconds = jest.fn();
    mockPlaybackState.mockReturnValueOnce('Idle');

    mockUseScrubControls.mockImplementationOnce(() => ({
      subscribeToScrubbingPositionSeconds:
        mockSubscribeToScrubbingPositionSeconds,
    }));
    const { wrapper } = setup();

    const componentScrubHandler =
      mockSubscribeToScrubbingPositionSeconds.mock.calls[0][0];

    act(() => {
      componentScrubHandler(undefined);
    });

    expect(wrapper.find(ScPausedIndicator).props()['$hide']).toEqual(false);
  });

  it('calls togglePlay when clicked', () => {
    const mockTogglePlay = jest.fn();
    mockUseScrubControls.mockImplementationOnce(() => ({
      subscribeToScrubbingPositionSeconds: jest.fn(() => () => undefined),
      togglePlay: mockTogglePlay,
    }));

    const { wrapper } = setup();

    expect(mockTogglePlay).not.toHaveBeenCalled();

    act(() => {
      wrapper.find(ScPlayPauseButton).simulate('click');
    });

    expect(mockTogglePlay).toHaveBeenCalledTimes(1);
  });

  it('has pause aria-label when player state is playing', () => {
    mockPlaybackState.mockReturnValueOnce('Playing');
    const { wrapper } = setup();

    expect(wrapper.find(ScPlayPauseButton).props()['aria-label']).toEqual(
      'Pause',
    );
  });

  it('has play aria-label when player state not is playing', () => {
    mockPlaybackState.mockReturnValue('Idle');
    const { wrapper } = setup();

    expect(wrapper.find(ScPlayPauseButton).props()['aria-label']).toEqual(
      'Play',
    );
  });
});
