import type { PlaybackState } from 'pulsar';
import { usePlaybackState, usePlayerController } from 'pulsar';
import type { FC } from 'react';
import { useCallback, useEffect, useRef } from 'react';
import { logger } from 'tachyon-logger';
import { usePrevious, useTimeout, useUnmount } from 'tachyon-utils';
import { useNativeAppProxy } from '../../../framework';
import { useScrubControls } from '../ScrubControls';

type SimpleFunc = () => void;
type SeekFunc = (positionSeconds: number) => void;

/**
 * The interface for calls from native into the app to manipulate video playback.
 */
export type MediaSessionProxy = {
  fastForward: SimpleFunc;
  pause: SimpleFunc;
  play: SimpleFunc;
  rewind: SimpleFunc;
  seekTo: SeekFunc;
  stop: SimpleFunc;
};

declare global {
  interface Window {
    mediaSessionProxy: MediaSessionProxy | undefined;
  }
}

type CreateMediaSessionProxyOpts = {
  pause: SimpleFunc | undefined;
  play: SimpleFunc | undefined;
  scrubAhead: SimpleFunc | undefined;
  scrubBack: SimpleFunc | undefined;
  scrubCommit: SimpleFunc | undefined;
  scrubTo: SeekFunc | undefined;
  stop: SimpleFunc | undefined;
};

function createMediaSessionProxy({
  pause,
  play,
  scrubAhead,
  scrubBack,
  scrubCommit,
  scrubTo,
  stop,
}: CreateMediaSessionProxyOpts): MediaSessionProxy {
  return {
    fastForward: () => {
      if (scrubCommit) {
        scrubAhead?.();
      }
    },
    pause: () => {
      pause?.();
    },
    play: () => {
      play?.();
    },
    rewind: () => {
      if (scrubCommit) {
        scrubBack?.();
      }
    },
    seekTo: (positionSeconds: number) => {
      if (scrubTo && scrubCommit) {
        scrubTo(positionSeconds);
        scrubCommit();
      }
    },
    stop: () => {
      stop?.();
    },
  };
}

/**
 * Component that binds the player to the native interface.
 *
 * NOTE: Currently, this is not hooked up to live streams.  It is only intended
 * to work on VODs and clips.  If/when the live channel player supports pausing
 * we can look into hooking this up there.
 */
export const MediaSession: FC = ({ children }) => {
  const { setPlaybackPosition, videoPlayerStateChanged } = useNativeAppProxy();
  const playbackState = usePlaybackState();
  const playerController = usePlayerController();
  const {
    scrubAhead,
    scrubBack,
    scrubCommit,
    scrubTo,
    subscribeToScrubbingPositionSeconds,
  } = useScrubControls();
  const isScrubbing = useRef(false);

  useEffect(() => {
    return subscribeToScrubbingPositionSeconds((nextPosition) => {
      isScrubbing.current = nextPosition !== undefined;
    });
  }, [subscribeToScrubbingPositionSeconds]);

  // Track previous state to avoid sending duplicate state change events
  const previousPlaybackState = usePrevious(playbackState);

  const handleVideoPlayerState = (state: PlaybackState) => {
    if (state !== previousPlaybackState) {
      logger.debug({
        category: 'MediaSession',
        message: `Reporting player state to native: ${state}`,
        package: 'starshot',
      });
      videoPlayerStateChanged(state);
    }
  };

  const getPosition = playerController?.getPosition;
  const playPromise = playerController?.play;
  const play = useCallback(() => {
    if (playPromise) {
      if (isScrubbing.current) {
        scrubCommit();
      } else {
        playPromise();
      }
    }
  }, [playPromise, scrubCommit]);

  const pauseOriginal = playerController?.pause;
  const pause = useCallback(() => {
    if (pauseOriginal) {
      if (isScrubbing.current) {
        scrubCommit();
      } else {
        pauseOriginal();
      }
    }
  }, [pauseOriginal, scrubCommit]);

  // TODO: Do we want to implement stop differently?  Should it navigate backward away from the video?
  const stop = pause;

  // Hook up the ability for native to control the player state
  useEffect(() => {
    const proxy = createMediaSessionProxy({
      pause,
      play,
      scrubAhead,
      scrubBack,
      scrubCommit,
      scrubTo,
      stop,
    });

    window.mediaSessionProxy = proxy;

    return () => {
      // NOTE: This check ensures that the proxy registration isn't removed by an old
      // MediaSession instance when a newer one may have been mounted before the old
      // one is unmounted.
      if (window.mediaSessionProxy === proxy) {
        window.mediaSessionProxy = undefined;
      }
    };
  }, [play, pause, stop, scrubAhead, scrubBack, scrubTo, scrubCommit]);

  // Doing this manually seems unnecessary, can we just set this with `usePlaybackTime`?
  useTimeout(() => {
    if (getPosition) {
      setPlaybackPosition(playbackState, getPosition() * 1000);
    }
  }, 1000);

  // Notify when the player has been unloaded
  useUnmount(() => {
    handleVideoPlayerState('Ended');
  });

  handleVideoPlayerState(playbackState);

  return <>{children}</>;
};

MediaSession.displayName = 'MediaSession';
