import { usePlayerController } from 'pulsar';
import { act } from 'react-dom/test-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { useNativeAppProxy } from '../../../framework';
import { useScrubControls } from '../ScrubControls';
import { MediaSession } from '.';

jest.mock('tachyon-logger', () => ({ logger: { debug: jest.fn() } }));

jest.mock('../../../framework', () => ({
  ...jest.requireActual('../../../framework'),
  useNativeAppProxy: jest.fn(),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerController: jest.fn(),
}));

jest.mock('../ScrubControls', () => ({
  ...jest.requireActual('../ScrubControls'),
  useScrubControls: jest.fn(),
}));

const mockUseNativeAppProxy = useNativeAppProxy as jest.Mock;
const mockUsePlayerController = usePlayerController as jest.Mock;
const mockUseScrubControls = useScrubControls as jest.Mock;

describe(MediaSession, () => {
  const setup = createMountWrapperFactory(MediaSession);
  const setupMocks = () => {
    mockUsePlayerController.mockImplementation(() => ({
      getPosition: jest.fn(),
      pause: jest.fn(),
      play: jest.fn(),
    }));
    mockUseScrubControls.mockImplementation(() => ({
      subscribeToScrubbingPositionSeconds: jest.fn(() => () => undefined),
    }));
    mockUseNativeAppProxy.mockImplementation(() => ({
      setPlaybackPosition: jest.fn(),
      videoPlayerStateChanged: jest.fn(),
    }));
  };

  beforeEach(() => {
    jest.resetAllMocks();
    setupMocks();
  });

  afterEach(() => {
    jest.restoreAllMocks();
  });

  it('adds and removes mediaSessionProxy to the window', () => {
    expect(window.mediaSessionProxy).toBeFalsy();
    const { wrapper } = setup();
    expect(window.mediaSessionProxy).toBeTruthy();
    act(() => {
      wrapper.unmount();
    });
    expect(window.mediaSessionProxy).toBeFalsy();
  });

  it('does not set timeout if getPosition is undefined', () => {
    const setPlaybackPosition = jest.fn();
    mockUsePlayerController.mockImplementationOnce(() => ({
      getPosition: undefined,
    }));
    mockUseNativeAppProxy.mockImplementationOnce(() => ({
      setPlaybackPosition,
    }));
    jest.useFakeTimers();
    setup();
    jest.advanceTimersByTime(1000);
    expect(setPlaybackPosition).not.toHaveBeenCalled();
  });

  it('periodically updates playback position for native app', () => {
    const setPlaybackPosition = jest.fn();
    const videoPlayerStateChanged = jest.fn();
    mockUseNativeAppProxy.mockImplementationOnce(() => ({
      setPlaybackPosition,
      videoPlayerStateChanged,
    }));
    jest.useFakeTimers();
    setup();
    jest.advanceTimersByTime(1000);
    expect(setPlaybackPosition).toHaveBeenCalledTimes(1);
  });

  it('calls relevant functions when prompted by the native app', () => {
    const getPosition = jest.fn();
    const play = jest.fn();
    const pause = jest.fn();
    mockUsePlayerController.mockImplementationOnce(() => ({
      getPosition,
      pause,
      play,
    }));
    const scrubAhead = jest.fn();
    const scrubBack = jest.fn();
    const scrubCommit = jest.fn();
    const scrubTo = jest.fn();
    const subscribeToScrubbingPositionSeconds = jest.fn(() => () => undefined);
    mockUseScrubControls.mockImplementationOnce(() => ({
      scrubAhead,
      scrubBack,
      scrubCommit,
      scrubTo,
      subscribeToScrubbingPositionSeconds,
    }));
    setup();
    window.mediaSessionProxy?.play();
    expect(play).toHaveBeenCalledTimes(1);
    window.mediaSessionProxy?.pause();
    expect(pause).toHaveBeenCalledTimes(1);
    window.mediaSessionProxy?.stop();
    expect(pause).toHaveBeenCalledTimes(2);
    window.mediaSessionProxy?.rewind();
    expect(scrubBack).toHaveBeenCalledTimes(1);
    window.mediaSessionProxy?.fastForward();
    expect(scrubAhead).toHaveBeenCalledTimes(1);
    expect(scrubCommit).not.toHaveBeenCalled();
    window.mediaSessionProxy?.seekTo(1000);
    expect(scrubTo).toHaveBeenCalledTimes(1);
    expect(scrubCommit).toHaveBeenCalledTimes(1);
  });

  it('calls relevant functions when prompted by the native app with scrubbingPosition', () => {
    const play = jest.fn();
    const pause = jest.fn();
    mockUsePlayerController.mockImplementationOnce(() => ({
      pause,
      play,
    }));
    const mockScrubCommit = jest.fn();
    const mockSubscribeToScrubbingPositionSeconds = jest.fn();
    mockUseScrubControls.mockImplementationOnce(() => ({
      scrubCommit: mockScrubCommit,
      subscribeToScrubbingPositionSeconds:
        mockSubscribeToScrubbingPositionSeconds,
    }));

    setup();

    const componentScrubHandler =
      mockSubscribeToScrubbingPositionSeconds.mock.calls[0][0];

    act(() => {
      componentScrubHandler(200);
    });

    window.mediaSessionProxy?.play();
    expect(play).not.toHaveBeenCalled();
    expect(mockScrubCommit).toHaveBeenCalledTimes(1);
    window.mediaSessionProxy?.pause();
    expect(pause).not.toHaveBeenCalled();
    expect(mockScrubCommit).toHaveBeenCalledTimes(2);
  });

  it('updates video player state', () => {
    const videoPlayerStateChanged = jest.fn();
    mockUseNativeAppProxy.mockImplementationOnce(() => ({
      videoPlayerStateChanged,
    }));
    const { wrapper } = setup();
    expect(videoPlayerStateChanged).not.toHaveBeenCalled();
    act(() => {
      wrapper.unmount();
    });
    expect(videoPlayerStateChanged).toHaveBeenCalledWith('Ended');
  });
});
