import { usePlaybackTime, usePlayerController } from 'pulsar';
import type { ChangeEvent, FC } from 'react';
import { useEffect, useMemo, useRef, useState } from 'react';
import styled from 'styled-components';
import { NodeNav, useFocus } from 'tachyon-tv-nav';
import {
  KeyValue,
  TVCodes,
  formatVideoLength,
  legacyKeyboardEventHandler,
} from 'tachyon-utils';
import { CoreText, Display, Layout } from 'twitch-core-ui';
import { accessiblyHidden } from '../../../common';
import { useScrubControls } from '../ScrubControls';
import { PlayerHead } from './PlayerHead';
import { ProgressBar } from './ProgressBar';
import { TOTAL_HEIGHT_REM } from './ui';
import { getPercentagePositions } from './utils';

/**
 * A hidden range input is used to power the seek bar behavior for all of the nice user input management for clicking,
 * dragging, accessibility, etc, but render a completely custom UI over top of it.
 *
 * The custom UI allows us to provide more information, as well as avoid the need
 * to do browser specific pseudo CSS selectors to properly style the input range
 * to our liking.
 */
const ScHiddenRangeInput = styled.input`
  cursor: pointer;
  height: 100%;
  ${accessiblyHidden}
  position: absolute;
  width: 100%;
`;

const ScSeekBarContainer = styled.div`
  height: ${TOTAL_HEIGHT_REM}rem;
  position: relative;
  width: 100%;
`;

type PlaybackScrubBarProps = {
  focusIndex: number;
};

// istanbul ignore next: trivial
export const PlaybackScrubBar: FC<PlaybackScrubBarProps> = ({ focusIndex }) => {
  const { focused, takeFocus } = useFocus(focusIndex);
  const controller = usePlayerController();
  const scrubbingPositionSecondsRef = useRef<number>();

  const {
    scrubAbort,
    scrubAhead,
    scrubBack,
    scrubCommit,
    subscribeToScrubbingPositionSeconds,
    togglePlay,
  } = useScrubControls();

  useEffect(() => {
    const updateScrubbingPosition = (
      nextScrubbingPosition: number | undefined,
    ) => {
      scrubbingPositionSecondsRef.current = nextScrubbingPosition;
    };

    return subscribeToScrubbingPositionSeconds(updateScrubbingPosition);
  }, [subscribeToScrubbingPositionSeconds]);

  useEffect(() => {
    if (!focused) {
      return;
    }

    const keydownHandler = legacyKeyboardEventHandler(
      (event: KeyboardEvent) => {
        if (event.key === KeyValue.Enter) {
          if (scrubbingPositionSecondsRef.current !== undefined) {
            scrubCommit();
          } else {
            togglePlay();
          }
        }
      },
    );

    document.addEventListener('keydown', keydownHandler);

    return () => {
      document.removeEventListener('keydown', keydownHandler);
    };
  }, [focused, scrubCommit, togglePlay]);

  useEffect(() => {
    const keydownHandler = legacyKeyboardEventHandler(
      (event: KeyboardEvent) => {
        if (!(event.key in TVCodes)) {
          return;
        }
        if (!focused) {
          takeFocus();
        }
        switch (event.key) {
          case KeyValue.PauseTV:
          case KeyValue.PlayTV:
            togglePlay();
            break;
          case KeyValue.NextTV:
            scrubAhead();
            break;
          case KeyValue.PrevTV:
            scrubBack();
            break;
        }
      },
    );

    document.addEventListener('keydown', keydownHandler);

    return () => {
      document.removeEventListener('keydown', keydownHandler);
    };
  }, [focused, scrubAhead, scrubBack, takeFocus, togglePlay]);

  if (!controller) {
    return null;
  }

  // convert 0 to undefined as getDuration will return 0 prior to initialization
  const totalVideoTimeSeconds =
    Math.floor(controller.getDuration()) || undefined;

  return (
    <NodeNav
      focusIndex={focusIndex}
      onDown={scrubAbort}
      onLeft={scrubBack}
      onRight={scrubAhead}
      onUp={scrubAbort}
    >
      <Layout display={Display.Flex} fullWidth onMouseEnter={takeFocus}>
        <ScSeekBarContainer>
          <PlaybackScrubBarContents
            focused={focused}
            totalVideoTimeSeconds={totalVideoTimeSeconds}
          />
        </ScSeekBarContainer>
        <Layout padding={{ left: 2 }}>
          {totalVideoTimeSeconds && (
            <CoreText>{formatVideoLength(totalVideoTimeSeconds)}</CoreText>
          )}
        </Layout>
      </Layout>
    </NodeNav>
  );
};

PlaybackScrubBar.displayName = 'PlaybackScrubBar';

type PlaybackScrubBarContentsProps = {
  focused: boolean;
  totalVideoTimeSeconds: number | undefined;
};

export const PlaybackScrubBarContents: FC<PlaybackScrubBarContentsProps> = ({
  focused,
  totalVideoTimeSeconds,
}) => {
  const { scrubCommit, scrubTo, subscribeToScrubbingPositionSeconds } =
    useScrubControls();
  const currentPlaybackPositionSeconds = usePlaybackTime();
  const [scrubbingPositionSeconds, setScrubbingPositionSeconds] = useState<
    number | undefined
  >();

  // We involve the scrubbing system for this because of the click and drag interaction.
  // In that case, the user needs a scrubbing experience with preview images.
  // Seeking eventually happens as a result of a scrubCommit from an onMouseUp listener
  // added to the ScHiddenRangeInput.
  // istanbul ignore next: trivial
  const progressBarOnChange = useMemo(
    () => (event: ChangeEvent<HTMLInputElement>) => {
      scrubTo(event.currentTarget.valueAsNumber);
    },
    [scrubTo],
  );

  // istanbul ignore next: trivial
  useEffect(() => {
    const updateScrubbingPosition = (
      nextScrubbingPosition: number | undefined,
    ) => {
      setScrubbingPositionSeconds(nextScrubbingPosition);
    };

    return subscribeToScrubbingPositionSeconds(updateScrubbingPosition);
  }, [subscribeToScrubbingPositionSeconds]);

  // istanbul ignore next: trivial
  const playHeadPositionSeconds =
    scrubbingPositionSeconds ?? currentPlaybackPositionSeconds;

  // istanbul ignore next: trivial
  const { currentPositionPercent, playHeadPositionPercent } =
    getPercentagePositions(
      totalVideoTimeSeconds,
      currentPlaybackPositionSeconds,
      playHeadPositionSeconds,
    );

  return (
    <>
      <ProgressBar currentPositionPercent={currentPositionPercent} />
      <PlayerHead
        focused={focused}
        isAccelerating={scrubbingPositionSeconds !== undefined}
        playerHeadOffset={playHeadPositionPercent}
        playerHeadSeconds={playHeadPositionSeconds}
      />
      <ScHiddenRangeInput
        max={totalVideoTimeSeconds}
        min={0}
        onChange={progressBarOnChange}
        onMouseUp={scrubCommit}
        step="1"
        type="range"
        value={playHeadPositionSeconds}
      />
    </>
  );
};

PlaybackScrubBarContents.displayName = 'PlaybackScrubBarContents';
