import { usePlayerController } from 'pulsar';
import type { FC } from 'react';
import { createContext, useContext, useEffect, useMemo, useRef } from 'react';
import { useRecentUserInput } from '../../../framework';
import { useScrubAcceleration } from './useScrubAcceleration';

export const DEFAULT_SCRUB_INTERVAL_SECONDS = 10;

type ScrubbingPositionSeconds = number | undefined;

export type ScrubControls = {
  scrubAbort: () => void;
  scrubAhead: () => void;
  scrubBack: () => void;
  scrubCommit: () => void;
  scrubTo: (positionSeconds: number) => void;
  subscribeToScrubbingPositionSeconds: (
    callback: (nextPosition: ScrubbingPositionSeconds) => void,
  ) => () => void;
  togglePlay: () => void;
};

// istanbul ignore next: trivial
const scrubControlsNoOp = {
  scrubAbort: (): void => undefined,
  scrubAhead: (): void => undefined,
  scrubBack: (): void => undefined,
  scrubCommit: (): void => undefined,
  scrubTo: (): void => undefined,
  subscribeToScrubbingPositionSeconds: () => () => undefined,
  togglePlay: (): void => undefined,
};

// istanbul ignore next: trivial
const scrubControlsContext = createContext<ScrubControls>(scrubControlsNoOp);

export const useScrubControls = (): ScrubControls =>
  useContext(scrubControlsContext);

type ScrubControlsRootProps = {
  scrubIntervalSeconds?: number;
};

/**
 * Root context for scrub controls. This component should be excluded
 * from any path that re-renders periodically (due to playback state or
 * playback time for example). This must be rendered inside of
 * {@link https://git.xarth.tv/pages/emerging-platforms/tachyon/d/packages/pulsar/player-core/#advanced-interactions|PlayerControllerRoot}.
 */
export const ScrubControlsRoot: FC<ScrubControlsRootProps> = ({
  children,
  scrubIntervalSeconds,
}) => {
  const controller = usePlayerController();
  const { cancelPersistedRecentUserInput, persistRecentUserInputIndefinitely } =
    useRecentUserInput();
  const totalVideoTimeSeconds = Math.floor(controller?.getDuration() ?? 0);

  const {
    accelerate: accelerateForward,
    resetAcceleration: resetForwardAcceleration,
  } = useScrubAcceleration({
    minSpeed: scrubIntervalSeconds ?? DEFAULT_SCRUB_INTERVAL_SECONDS,
    range: totalVideoTimeSeconds,
    upperBound: totalVideoTimeSeconds,
  });

  const {
    accelerate: accelerateBack,
    resetAcceleration: resetBackAcceleration,
  } = useScrubAcceleration({
    minSpeed: scrubIntervalSeconds ?? DEFAULT_SCRUB_INTERVAL_SECONDS,
    range: totalVideoTimeSeconds,
    upperBound: 0,
  });

  const scrubbingPositionSeconds = useRef<ScrubbingPositionSeconds>();
  const scrubListeners =
    useRef<
      Set<(newScrubbingPositionSeconds: ScrubbingPositionSeconds) => void>
    >();

  const setScrubbingPositionSeconds = (
    newPosition: ScrubbingPositionSeconds,
  ) => {
    scrubbingPositionSeconds.current = newPosition;
    scrubListeners.current?.forEach((callback) => {
      callback(newPosition);
    });
  };

  // When we get a keyup event clear any acceleration that might have happened
  useEffect(() => {
    const resetAcceleration = () => {
      resetForwardAcceleration();
      resetBackAcceleration();
    };
    document.addEventListener('keyup', resetAcceleration);
    return () => {
      document.removeEventListener('keyup', resetAcceleration);
    };
  }, [resetForwardAcceleration, resetBackAcceleration]);

  const value = useMemo(() => {
    if (!controller) {
      // istanbul ignore next: trivial
      return scrubControlsNoOp;
    }
    return {
      scrubAbort: () => {
        setScrubbingPositionSeconds(undefined);
        cancelPersistedRecentUserInput();
        controller.play();
      },
      scrubAhead: () => {
        controller.pause();
        persistRecentUserInputIndefinitely();
        setScrubbingPositionSeconds(
          accelerateForward(
            scrubbingPositionSeconds.current ??
              Math.floor(controller.getPosition()),
          ),
        );
      },
      scrubBack: () => {
        controller.pause();
        persistRecentUserInputIndefinitely();
        setScrubbingPositionSeconds(
          accelerateBack(
            scrubbingPositionSeconds.current ??
              Math.floor(controller.getPosition()),
          ),
        );
      },
      scrubCommit: () => {
        if (scrubbingPositionSeconds.current === undefined) {
          return;
        }
        cancelPersistedRecentUserInput();
        controller.seekTo(scrubbingPositionSeconds.current);
        controller.play().then(() => {
          setScrubbingPositionSeconds(undefined);
        });
      },
      scrubTo: (positionSeconds: number) => {
        controller.pause();
        const sanitizedPosition = Math.min(
          Math.max(0, Math.floor(positionSeconds)),
          totalVideoTimeSeconds,
        );
        setScrubbingPositionSeconds(sanitizedPosition);
      },
      subscribeToScrubbingPositionSeconds: (
        callback: (
          newScrubbingPositionSeconds: ScrubbingPositionSeconds,
        ) => void,
      ) => {
        scrubListeners.current ??= new Set();
        scrubListeners.current.add(callback);
        return () => {
          scrubListeners.current?.delete(callback);
        };
      },
      togglePlay: () => {
        if (controller.isPaused()) {
          controller.play();
        } else {
          controller.pause();
        }
      },
    };
  }, [
    accelerateBack,
    accelerateForward,
    cancelPersistedRecentUserInput,
    controller,
    persistRecentUserInputIndefinitely,
    totalVideoTimeSeconds,
  ]);

  return <scrubControlsContext.Provider children={children} value={value} />;
};

ScrubControlsRoot.displayName = 'ScrubControlsRoot';
