import { act, renderHook } from '@testing-library/react-hooks';
import { datatype } from 'faker';
import { mockPlayerController, usePlayerController } from 'pulsar';
import {
  DEFAULT_SCRUB_INTERVAL_SECONDS,
  ScrubControlsRoot,
  useScrubControls,
} from '.';

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

describe(ScrubControlsRoot, () => {
  const mockIsPaused = jest.fn(() => false);
  const mockPause = jest.fn();
  const mockPlay = jest.fn();
  let currentTime = 0;
  let currentScrubbingTime: number | undefined;
  const contentDuration = 10000;
  const subscribeToScrubbingPosCb = (newScrubbingTime: number | undefined) => {
    currentScrubbingTime = newScrubbingTime;
  };

  function setup() {
    return renderHook(useScrubControls, {
      wrapper: ScrubControlsRoot,
    });
  }
  beforeEach(() => {
    currentTime = 0;
    currentScrubbingTime = undefined;
    mockPlay.mockReset();
    mockPlay.mockImplementation(() => Promise.resolve());
    mockPause.mockReset();
    mockUsePlayerController.mockReturnValue(
      mockPlayerController({
        getDuration: () => contentDuration,
        getPosition: () => currentTime,
        isPaused: mockIsPaused,
        pause: mockPause,
        play: mockPlay,
        seekTo: jest.fn(),
      }),
    );
  });

  describe('scrubAbort', () => {
    it('calls play and sets scrubbingPostitionSeconds to undefined', () => {
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      expect(mockPlay).not.toHaveBeenCalled();

      act(() => {
        result.current.scrubAhead();
      });
      act(() => {
        result.current.scrubAbort();
      });

      expect(mockPlay).toHaveBeenCalledTimes(1);
      expect(currentScrubbingTime).toBeUndefined();
    });
  });

  describe('scrubAhead', () => {
    it('pauses the controller when called', () => {
      const { result } = setup();
      expect(mockUsePlayerController().pause).not.toHaveBeenCalled();
      act(() => {
        result.current.scrubAhead();
      });
      expect(mockUsePlayerController().pause).toHaveBeenCalledTimes(1);
    });

    it('sets the scrubbingPostitionSeconds and increments by the skip ahead interval when called', () => {
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      expect(currentScrubbingTime).toBeUndefined();
      act(() => {
        result.current.scrubAhead();
      });
      expect(currentScrubbingTime).toBe(
        currentTime + DEFAULT_SCRUB_INTERVAL_SECONDS,
      );
    });

    it('sets the scrubbingPostitionSeconds to the total content duration when scrub ahead over shoots', () => {
      currentTime = contentDuration - DEFAULT_SCRUB_INTERVAL_SECONDS / 2;
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      act(() => {
        result.current.scrubAhead();
      });
      expect(currentScrubbingTime).toBe(contentDuration);
    });
  });

  describe('scrubBack', () => {
    it('pauses the controller when called', () => {
      const { result } = setup();
      expect(mockUsePlayerController().pause).not.toHaveBeenCalled();
      act(() => {
        result.current.scrubBack();
      });
      expect(mockUsePlayerController().pause).toHaveBeenCalledTimes(1);
    });

    it('sets the scrubbingPostitionSeconds and decrements by the skip back interval when called', () => {
      currentTime = contentDuration;
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      act(() => {
        result.current.scrubBack();
      });
      expect(currentScrubbingTime).toBe(
        currentTime - DEFAULT_SCRUB_INTERVAL_SECONDS,
      );
    });

    it('sets the scrubbingPostitionSeconds to 0 when scrub back over shoots', () => {
      currentTime = 0 + DEFAULT_SCRUB_INTERVAL_SECONDS / 2;
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      act(() => {
        result.current.scrubBack();
      });
      expect(currentScrubbingTime).toBe(0);
    });
  });

  describe('scrubCommit', () => {
    it('seeks to the scrubbingPostitionSeconds, plays, and resets scrubbingPostitionSeconds when scrubbingPostitionSeconds is defined', async () => {
      const { result } = setup();
      act(() => {
        result.current.scrubAhead();
      });

      const controller = mockUsePlayerController();
      expect(controller.play).not.toHaveBeenCalled();
      expect(controller.seekTo).not.toHaveBeenCalled();

      await act(async () => {
        result.current.scrubCommit();
        // wait for the mocked play function promise to resolve
        await controller.play();
      });

      expect(controller.play).toHaveBeenCalled();
      expect(controller.seekTo).toHaveBeenCalledWith(
        DEFAULT_SCRUB_INTERVAL_SECONDS,
      );
    });

    it('no-ops when scrubbingPositionSeconds is undefined', () => {
      const { result } = setup();
      act(() => {
        result.current.scrubCommit();
      });

      const controller = mockUsePlayerController();

      expect(controller.seekTo).not.toHaveBeenCalled();
      expect(controller.play).not.toHaveBeenCalled();
    });
  });

  describe('scrubTo', () => {
    it('pauses the controller when called', () => {
      const { result } = setup();
      expect(mockUsePlayerController().pause).not.toHaveBeenCalled();
      act(() => {
        result.current.scrubTo(DEFAULT_SCRUB_INTERVAL_SECONDS);
      });
      expect(mockUsePlayerController().pause).toHaveBeenCalledTimes(1);
    });

    it('scrubs to the specified position when it is in bounds', () => {
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      expect(currentScrubbingTime).toBeUndefined();
      act(() => {
        result.current.scrubTo(DEFAULT_SCRUB_INTERVAL_SECONDS);
      });
      expect(currentScrubbingTime).toBe(DEFAULT_SCRUB_INTERVAL_SECONDS);
    });

    it('scrubs to the end of the video when the specified position is greater than the duration', () => {
      currentTime = contentDuration - DEFAULT_SCRUB_INTERVAL_SECONDS / 2;
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      act(() => {
        result.current.scrubTo(currentTime + DEFAULT_SCRUB_INTERVAL_SECONDS);
      });
      expect(currentScrubbingTime).toBe(contentDuration);
    });

    it('rounds the input down if it is a decimal', () => {
      const decimalNumber = datatype.float({ max: contentDuration });
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      act(() => {
        result.current.scrubTo(decimalNumber);
      });
      expect(currentScrubbingTime).toBe(Math.floor(decimalNumber));
    });

    it('sets scrubbing player head to 0 when a negative number is provided', () => {
      const negativeNumber = datatype.number(contentDuration) * -1;
      const { result } = setup();
      result.current.subscribeToScrubbingPositionSeconds(
        subscribeToScrubbingPosCb,
      );
      act(() => {
        result.current.scrubTo(negativeNumber);
      });
      expect(currentScrubbingTime).toBe(0);
    });
  });

  describe('togglePlay', () => {
    it('calls play when the player is paused', () => {
      const { result } = setup();
      mockIsPaused.mockReturnValueOnce(true);
      expect(mockPlay).not.toHaveBeenCalled();
      act(() => {
        result.current.togglePlay();
      });

      expect(mockPlay).toHaveBeenCalledTimes(1);
    });

    it('calls pause when the player is playing', () => {
      const { result } = setup();
      mockIsPaused.mockReturnValueOnce(false);
      expect(mockPause).not.toHaveBeenCalled();
      act(() => {
        result.current.togglePlay();
      });

      expect(mockPause).toHaveBeenCalledTimes(1);
    });
  });

  describe('subscribeToScrubbingPositionSeconds', () => {
    it('adds and removes subscribers', () => {
      const { result } = setup();
      const listenerOne = jest.fn();
      const listenerTwo = jest.fn();

      const removeListenerOne =
        result.current.subscribeToScrubbingPositionSeconds(listenerOne);
      result.current.subscribeToScrubbingPositionSeconds(listenerTwo);

      act(() => {
        result.current.scrubAhead();
      });

      expect(listenerOne).toHaveBeenCalledTimes(1);
      expect(listenerTwo).toHaveBeenCalledTimes(1);

      removeListenerOne();

      act(() => {
        result.current.scrubAhead();
      });

      expect(listenerOne).toHaveBeenCalledTimes(1);
      expect(listenerTwo).toHaveBeenCalledTimes(2);
    });
  });
});
