import { useMemo } from 'react';
import { quadratic } from './AccelerationFunctions';

export type UseScrubAccelerationProps = {
  minSpeed: number;
  range: number;
  upperBound: number;
};

type UseScrubAcceleration = {
  accelerate: (currentValue: number) => number;
  isAccelerating: () => boolean;
  resetAcceleration: () => void;
};

export const useScrubAcceleration = ({
  minSpeed,
  range,
  upperBound,
}: UseScrubAccelerationProps): UseScrubAcceleration => {
  return useMemo((): UseScrubAcceleration => {
    let currentGear = 0;
    const accelerationGears = quadratic(minSpeed, range);

    return {
      accelerate: (currentValue) => {
        const nextValue = clampSpeedOnUpperBound(
          currentValue,
          accelerationGears[currentGear],
          upperBound,
        );

        if (currentGear < accelerationGears.length - 1) {
          currentGear = currentGear + 1;
        }

        return nextValue;
      },
      isAccelerating: () => currentGear > 0,
      resetAcceleration: () => (currentGear = 0),
    };
  }, [range, upperBound, minSpeed]);
};

export function clampSpeedOnUpperBound(
  currentValue: number,
  increment: number,
  upperBound: number,
): number {
  // If we're decelerating increment will be greater than upperBound
  if (increment > upperBound) {
    return Math.max(currentValue - increment, upperBound);
  }
  return Math.min(currentValue + increment, upperBound);
}
