import type { PulsarProps } from 'pulsar';
import { PulsarCore, useMuted, usePlaybackState } from 'pulsar';
import type { FC } from 'react';
import { memo, useEffect, useMemo } from 'react';
import { useCurrentUser } from 'tachyon-auth';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { useCustomLatency } from 'tachyon-latency-tracker';
import { useConstCallback } from 'tachyon-utils';
import { AlignItems, Display, JustifyContent, Layout } from 'twitch-core-ui';
import { CLIENT_ID } from '../../../../config';
import { useStarshotVisualDebugMode } from '../../../framework';
import { useBackgroundVideoPauser } from './useBackgroundVideoPauser';

export type StarshotPulsarPlayerCoreProps = PulsarProps;

const StarshotPulsarPlayerCoreBase: FC<StarshotPulsarPlayerCoreProps> = (
  pulsarProps,
) => {
  const { client, common } = useStaticEnvironment();
  const { debugModeActive } = useStarshotVisualDebugMode();
  const { getAuthToken } = useCurrentUser();

  const reportLatencyEvent = useCustomLatency();
  const reportPlayerReady = useConstCallback(reportLatencyEvent('playerReady'));

  const pulsarEnv = useMemo(
    () => ({
      authToken: getAuthToken(),
      benchmarkSessionID: client.sessionID,
      clientAPIID: CLIENT_ID,
      clientApp: common.clientApp,
      clientBuildID: common.appVersion,
      deviceID: client.deviceID,
      platform: common.platform,
    }),
    [
      client.deviceID,
      client.sessionID,
      common.appVersion,
      common.clientApp,
      common.platform,
      getAuthToken,
    ],
  );

  // in local dev for old Chromium browsers, we don't render the player since
  // the lack of codec support will cause it to crash
  if (
    common.appEnvironment === 'development' &&
    client.agentInfo.satisfies({ chrome: '<=60' })
  ) {
    return (
      <Layout
        alignItems={AlignItems.Center}
        display={Display.Flex}
        fullHeight
        fullWidth
        justifyContent={JustifyContent.Center}
      >
        Player placeholder for old Chromium
      </Layout>
    );
  }

  // LG currently crashes if a WASM is used for playback: https://jira.twitch.com/browse/EMP-2925
  const forceASM = common.platform === Platform.LG;
  return (
    <PulsarCore
      controls={false}
      debugMode={debugModeActive}
      environment={pulsarEnv}
      forceASM={forceASM}
      onFirstControlsAppear={reportPlayerReady}
      // Some AndroidTV devices have a default poster
      // so we pass in an invalid url to override the
      // default and instead show a black screen for
      // consistency with other platforms
      poster="none"
      {...pulsarProps}
    />
  );
};

StarshotPulsarPlayerCoreBase.displayName = 'StarshotPulsarPlayerCoreBase';

// Memoize so buffering state changes does not cause PulsarCore to rerender
const MemoStarshotPulsarPlayerCoreBase = memo(StarshotPulsarPlayerCoreBase);
MemoStarshotPulsarPlayerCoreBase.displayName =
  'MemoStarshotPulsarPlayerCoreBase';

export const StarshotPulsarPlayerCore: FC<StarshotPulsarPlayerCoreProps> = (
  props,
) => {
  const playbackState = usePlaybackState();
  const { muted, setMuted } = useMuted();
  const reportLatencyEvent = useCustomLatency();

  // This causes the component it is in to re-render so keep out of BasePlayerBase
  useBackgroundVideoPauser();

  useEffect(() => {
    if (playbackState === 'Playing') {
      reportLatencyEvent('playerPlaying')();
    }
  }, [playbackState, reportLatencyEvent]);

  // Player-Core will mute itself if it encounters errors when autoplay fails.
  // Re-unmute the player since Starshot we defer all volume / mute control to the device / OS
  // so users will have no way to unmute manually.
  useEffect(() => {
    if (!props.muted && playbackState === 'Playing' && muted) {
      setMuted(false);
    }
  }, [muted, playbackState, props.muted, setMuted]);

  return <MemoStarshotPulsarPlayerCoreBase {...props} />;
};

StarshotPulsarPlayerCore.displayName = 'StarshotPulsarPlayerCore';
