import { PulsarCore, useMuted, usePlaybackState } from 'pulsar';
import { act } from 'react-dom/test-utils';
import {
  Platform,
  createMockStaticEnvironmentContext,
  getOtherEnvMock,
} from 'tachyon-environment';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { StarshotPulsarPlayerCore } from '.';

jest.mock('tachyon-discovery', () => ({
  useDiscoveryTracking: () => ({ getAndResetPlayerTracking: jest.fn() }),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  // Jest's jsdom integration doesn't properly mock window.navigator.userAgent.
  // player-core calls Bowser internally with this value, causing tests to fail if this mock isn't present.
  PulsarCore: () => null,
  useMuted: jest.fn(),
  usePlaybackState: jest.fn(),
}));

const mockUseMuted = useMuted as jest.Mock;
const mockUsePlaybackState = usePlaybackState as jest.Mock;

describe(StarshotPulsarPlayerCore, () => {
  beforeEach(() => {
    mockUseMuted.mockReturnValue({ muted: false, setMuted: jest.fn() });
    mockUsePlaybackState.mockReturnValue('Ready');
  });

  const setup = createMountWrapperFactory(
    StarshotPulsarPlayerCore,
    () => ({
      loop: false,
      muted: false,
      src: 'https://threalderekt.m3u8',
    }),
    {
      wrappingContexts: () => [
        createMockStaticEnvironmentContext(
          getOtherEnvMock({ platform: Platform.StarshotWeb }),
        ),
      ],
    },
  );

  it('does not unmute itself when unmuted playback begins', () => {
    mockUseMuted.mockReturnValue({ muted: false, setMuted: jest.fn() });
    mockUsePlaybackState.mockReturnValue('Playing');

    act(() => {
      setup();
    });
    expect(useMuted().setMuted).not.toHaveBeenCalled();
  });

  it('unmutes itself whenever muted playback begins', () => {
    mockUseMuted.mockReturnValue({ muted: true, setMuted: jest.fn() });
    mockUsePlaybackState.mockReturnValue('Playing');

    act(() => {
      setup();
    });
    expect(useMuted().setMuted).toHaveBeenCalledWith(false);
  });

  it('does not unmute itself whenever muted playback begins intentionally', () => {
    mockUseMuted.mockReturnValue({ muted: true, setMuted: jest.fn() });
    mockUsePlaybackState.mockReturnValue('Playing');

    act(() => {
      setup({ muted: true });
    });
    expect(useMuted().setMuted).not.toHaveBeenCalled();
  });

  describe('ASM / WASM', () => {
    it('forces ASM for LG platforms', () => {
      const { wrapper } = setup(undefined, {
        wrappingContexts: [
          createMockStaticEnvironmentContext(
            getOtherEnvMock({ platform: Platform.LG }),
          ),
        ],
      });

      expect(wrapper.find(PulsarCore)).toHaveProp({ forceASM: true });
    });

    it('does not force ASM for non-LG platforms', () => {
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ forceASM: false });
    });
  });
});
