import dynamic from 'next/dynamic';
import type {
  PulsarBackend,
  PulsarHlsjsProps,
  PulsarNativeProps,
  PulsarProps,
} from 'pulsar';
import type { FC } from 'react';
import { exhaustedCase } from 'tachyon-utils';
import type { StarshotPulsarPlayerCoreProps } from './StarshotPulsarPlayerCore';

// istanbul ignore next: trivial
export const LazyStarshotPulsarPlayerCore =
  dynamic<StarshotPulsarPlayerCoreProps>(() =>
    import('./StarshotPulsarPlayerCore').then(
      (exports) => exports.StarshotPulsarPlayerCore,
    ),
  );

// istanbul ignore next: trivial
export const LazyStarshotPulsarNative = dynamic<PulsarNativeProps>(() =>
  import('pulsar').then((exports) => exports.PulsarNative),
);

// istanbul ignore next: trivial
export const LazyStarshotPulsarHlsjs = dynamic<PulsarHlsjsProps>(() =>
  import('pulsar').then((exports) => exports.PulsarHlsjs),
);

export type PulsarBackendLoaderProps = Omit<PulsarProps, 'volume'> & {
  backend: PulsarBackend;
};

/**
 * Handles switching between Pulsar backends. Uses Dynamic Imports
 * to avoid bundle size inflation.
 */
// istanbul ignore next: trivial
export const PulsarBackendLoader: FC<PulsarBackendLoaderProps> = ({
  backend,
  ...pulsarProps
}) => {
  // Volume is always max (1) since we delegate all volume control to the
  // TV remote / OS. Consumers of `PulsarBackendLoader` can set `muted={true}` if they want
  // video to play silently.
  switch (backend) {
    case 'hlsjs':
      return <LazyStarshotPulsarHlsjs {...pulsarProps} volume={1} />;
    case 'native':
      return <LazyStarshotPulsarNative {...pulsarProps} volume={1} />;
    case 'mediaplayer':
      return <LazyStarshotPulsarPlayerCore {...pulsarProps} volume={1} />;
    default:
      return exhaustedCase(
        backend,
        <LazyStarshotPulsarPlayerCore {...pulsarProps} volume={1} />,
      );
  }
};

PulsarBackendLoader.displayName = 'PulsarBackendLoader';
