import type { QualitySettings } from 'pulsar';
import { formatVideoQualityName } from 'pulsar';
import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import { SVGAsset } from 'twitch-core-ui';
import { useStarshotNotification } from '../../../hooks';
import {
  FlyOutMenu,
  FocusableIconButton,
  FocusableTextButton,
  useFlyOutControls,
} from '../../common';

type QualitySelectButtonProps = {
  focusIndex: number;
  qualitySettings: QualitySettings | null;
};

type QualitySelectFlyoutContentsProps = Pick<
  QualitySettings,
  'setAutoQuality' | 'setQuality'
> & {
  qualityOptions: QualitySettings['options'];
};

export const QualitySelectButton: FC<QualitySelectButtonProps> = ({
  focusIndex,
  qualitySettings,
}) => {
  const { formatMessage } = useIntl();
  const { publish } = useStarshotNotification();
  const buttonLabel = formatMessage('Quality Settings', 'QualitySelect');

  // render a placeholder until the player has loaded and qualities are available
  // in the very unlikely chance that there are no qualities, rather than disabling the button
  // which prevents our focus system from moving DOM focus to this button (causing unexpected side-effects)
  // we send the user a notification explaining what happened. This seemed easier than trying to create a separate
  // balloon system in a constrained space
  if (!qualitySettings || qualitySettings.options.length === 0) {
    return (
      <FocusableIconButton
        aria-label={buttonLabel}
        focusIndex={focusIndex}
        icon={SVGAsset.NavSettings}
        onClick={() => {
          publish({
            autoCloseMs: 5000,
            meta: {
              displayType: 'snack-bar',
              message: formatMessage(
                'No video quality options available',
                'QualitySelect',
              ),
            },
            type: 'info',
          });
        }}
      />
    );
  }

  const {
    isAutoQuality,
    options,
    quality: activeQuality,
    setAutoQuality,
    setQuality,
  } = qualitySettings;

  // filter out audio-only options that can be returned by the player
  const qualityOptions = options.filter((quality) => {
    return !['audio_only', 'audio only'].includes(
      formatVideoQualityName(quality).toLowerCase(),
    );
  });

  const currentQualityIndex = isAutoQuality
    ? 0
    : qualityOptions.findIndex((quality) => {
        return (
          formatVideoQualityName(quality) ===
          formatVideoQualityName(activeQuality)
        );
        // auto quality + the quality options
      }) + 1;

  return (
    <FlyOutMenu
      aria-label={buttonLabel}
      elementCount={qualityOptions.length + 1}
      focusIndex={focusIndex}
      icon={SVGAsset.NavSettings}
      initialChildFocusIndex={currentQualityIndex}
      interactionContent={'open_quality_menu'}
    >
      <FlyOutContents
        qualityOptions={qualityOptions}
        setAutoQuality={setAutoQuality}
        setQuality={setQuality}
      />
    </FlyOutMenu>
  );
};

const FlyOutContents: FC<QualitySelectFlyoutContentsProps> = ({
  qualityOptions,
  setAutoQuality,
  setQuality,
}) => {
  const { setShowFlyout } = useFlyOutControls();
  const { formatMessage } = useIntl();
  return (
    <>
      <FocusableTextButton
        focusIndex={0}
        interactionContent="set_auto_quality"
        onClick={() => {
          setAutoQuality(true);
          setShowFlyout(false);
        }}
      >
        {formatMessage('Auto', 'QualitySelect')}
      </FocusableTextButton>
      {qualityOptions.map((quality, index) => {
        const name = formatVideoQualityName(quality);
        // Auto quality is indexed at 0 above.
        const qualityOptionsFocusIndex = index + 1;

        return (
          <FocusableTextButton
            focusIndex={qualityOptionsFocusIndex}
            interactionContent="set_quality"
            key={name}
            onClick={() => {
              setQuality(quality);
              setShowFlyout(false);
            }}
          >
            {name}
          </FocusableTextButton>
        );
      })}
    </>
  );
};

QualitySelectButton.displayName = 'QualitySelectButton';
FlyOutContents.displayName = 'QualitySelectFlyOutContents';
