import type { PulsarCoreQuality, Quality } from 'pulsar';
import { act } from 'react-dom/test-utils';
import {
  createMockNotificationContext,
  notificationContext,
} from 'tachyon-notification';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { NodeNav } from 'tachyon-tv-nav';
import { FocusableIconButton, FocusableTextButton } from '../../common';
import { QualitySelectButton } from '.';

describe(QualitySelectButton, () => {
  const setup = createMountWrapperFactory(
    QualitySelectButton,
    () => {
      const options = [{ name: '720p' }, { name: '1080p' }] as Array<
        Quality<PulsarCoreQuality>
      >;

      return {
        focusIndex: 0,
        qualitySettings: {
          isAutoQuality: false,
          options,
          quality: options[0],
          setAutoQuality: jest.fn(),
          setQuality: jest.fn(),
        },
      };
    },
    {
      wrappingContexts: () => [
        createMockNotificationContext({ publish: jest.fn() }),
      ],
    },
  );

  it('emits a notification on click if qualitySettings are not available', () => {
    const { contexts, wrapper } = setup({ qualitySettings: null });
    expect(contexts.get(notificationContext).publish).not.toHaveBeenCalled();

    wrapper.find(FocusableIconButton).simulate('click');
    expect(contexts.get(notificationContext).publish).toHaveBeenCalledTimes(1);
  });

  it('emits a notification on click if no qualities are available', () => {
    const { contexts, wrapper } = setup({ qualitySettings: { options: [] } });
    expect(contexts.get(notificationContext).publish).not.toHaveBeenCalled();

    wrapper.find(FocusableIconButton).simulate('click');
    expect(contexts.get(notificationContext).publish).toHaveBeenCalledTimes(1);
  });

  it('renders a disabled button if the options array is empty', () => {
    const { wrapper } = setup({ qualitySettings: { options: [] } });
    expect(wrapper.find(FocusableIconButton)).toExist();
  });

  it('renders the flyout if valid options are present', () => {
    const { wrapper } = setup();

    wrapper.find(FocusableIconButton).simulate('click');
    expect(wrapper.find(FocusableTextButton)).toExist();
  });

  it('sets auto quality and dismisses after selecting the auto quality option', () => {
    const { props, wrapper } = setup();

    wrapper.find(FocusableIconButton).simulate('click');
    expect(wrapper.find(FocusableTextButton)).toExist();

    wrapper.find(FocusableTextButton).at(0).simulate('click');
    expect(wrapper.find(FocusableTextButton)).not.toExist();
    expect(props.qualitySettings?.setAutoQuality).toHaveBeenCalled();
  });

  it('sets quality and dismisses after selecting a quality', () => {
    const { props, wrapper } = setup();

    wrapper.find(FocusableIconButton).simulate('click');
    expect(wrapper.find(FocusableTextButton)).toExist();

    wrapper.find(FocusableTextButton).at(1).simulate('click');
    expect(wrapper.find(FocusableTextButton)).not.toExist();
    expect(props.qualitySettings?.setQuality).toHaveBeenCalled();
  });

  it('dismisses after navigating down out of the flyout', () => {
    const { props, wrapper } = setup();

    wrapper.find(FocusableIconButton).simulate('click');
    expect(wrapper.find(FocusableTextButton)).toExist();

    act(() => {
      wrapper.find(NodeNav).prop('onDown')!();
    });
    wrapper.update();

    expect(wrapper.find(FocusableTextButton)).not.toExist();
    expect(props.qualitySettings?.setQuality).not.toHaveBeenCalled();
  });
});
