import {
  PlayerState,
  setMatureCookie,
  usePlaybackRestrictions,
  usePlaybackState,
  usePlayerError,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect } from 'react';
import { graphql } from 'react-relay/hooks';
import type { OmitRefType } from 'tachyon-type-library';
import { exhaustedCase } from 'tachyon-utils';
import {
  GeoBlocked,
  MatureGate,
  SubOnlyLive,
  UnknownRestriction,
  VideoError,
} from '../Overlays';
import type { BaseRestrictionProps } from '../types';
import { EndOfStreamPrompt } from './EndOfStreamPrompt';
import type { ChannelRestriction_channel } from './__generated__/ChannelRestriction_channel.graphql';

// We define this fragment outside of a container since the values are all unmasked
// and also used by the restriction detection functions
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelRestriction_channel on User {
    id
    login
    displayName
    broadcastSettings {
      isMature
    }
    stream {
      restrictionType
      self {
        canWatch
      }
    }
    ...SubOnlyLive_channel @relay(mask: false)
  }
`;

export type ChannelRestrictionProps = BaseRestrictionProps & {
  channel: OmitRefType<ChannelRestriction_channel>;
};

/**
 * Component used for consistent presentation restrictions for stream content.
 * Requires `<PlayerControllerRoot>` and `<ChannelPlaybackRestrictionsRoot>` context providers.
 *
 * @example
 * <PlayerControllerRoot>
 *   <ChannelPlaybackRestrictionsRoot>
 *     <ChannelRestriction {...props}>
 *       {streamPlayer}
 *     </ChannelRestriction>
 *   </ChannelPlaybackRestrictionsRoot>
 * </PlayerControllerRoot>
 */
export const ChannelRestriction: FC<ChannelRestrictionProps> = ({
  channel,
  children,
  focusIndex,
  passThrough,
}) => {
  const playerError = usePlayerError();
  const playbackState = usePlaybackState();
  const {
    activeRestriction,
    addRestriction,
    handlePlayerError,
    removeRestriction,
  } = usePlaybackRestrictions();

  useEffect(() => {
    if (playerError) {
      handlePlayerError(playerError);
    }
  }, [handlePlayerError, playerError]);

  useEffect(() => {
    if (playbackState === PlayerState.ENDED) {
      addRestriction('playback-ended');
    }
  }, [addRestriction, playbackState]);

  if (passThrough) {
    return <>{children}</>;
  }
  switch (activeRestriction) {
    case 'playback-error':
      return (
        <VideoError
          focusIndex={focusIndex}
          onReloadClick={() => {
            removeRestriction('playback-error');
          }}
        />
      );
    case 'player-auth-geo':
      return <GeoBlocked focusIndex={focusIndex} />;
    case 'gql-mature':
      return (
        <MatureGate
          focusIndex={focusIndex}
          onAccept={() => {
            setMatureCookie();
            removeRestriction('gql-mature');
          }}
        />
      );
    case 'gql-sub-only':
      return <SubOnlyLive channel={channel} focusIndex={focusIndex} />;
    // TODO render a specific DRM error
    case 'player-auth-drm':
    case 'player-auth-vod-sub-only':
    case 'player-auth-unknown':
    case 'gql-vod-muted-segments':
      return <UnknownRestriction focusIndex={focusIndex} />;
    case 'playback-ended':
      return <EndOfStreamPrompt channel={channel} focusIndex={focusIndex} />;
    case null:
      return <>{children}</>;
    default: {
      return exhaustedCase(
        activeRestriction,
        <UnknownRestriction focusIndex={focusIndex} />,
      );
    }
  }
};
ChannelRestriction.displayName = 'ChannelRestriction';
