import {
  setMatureCookie,
  usePlaybackRestrictions,
  usePlayerError,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect } from 'react';
import { graphql } from 'react-relay/hooks';
import type { OmitRefType } from 'tachyon-type-library';
import { exhaustedCase } from 'tachyon-utils';
import {
  GeoBlocked,
  MatureGate,
  UnknownRestriction,
  VideoError,
} from '../Overlays';
import type { BaseRestrictionProps } from '../types';
import type { ClipRestriction_clip } from './__generated__/ClipRestriction_clip.graphql';

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ClipRestriction_clip on Clip {
    broadcaster {
      id
      broadcastSettings {
        isMature
      }
    }
  }
`;

export type ClipRestrictionProps = BaseRestrictionProps & {
  clip: OmitRefType<ClipRestriction_clip>;
};

/**
 * Component used for consistent presentation restrictions for clip content.
 * Requires `<PlayerControllerRoot>` and `<ClipPlaybackRestrictionsRoot>` context providers.
 *
 * @example
 * <PlayerControllerRoot>
 *   <ClipPlaybackRestrictionsRoot>
 *     <ClipRestriction {...props}>
 *       {clipPlayer}
 *     </ClipRestriction>
 *   </ClipPlaybackRestrictionsRoot>
 * </PlayerControllerRoot>
 */
export const ClipRestriction: FC<ClipRestrictionProps> = ({
  children,
  focusIndex,
  passThrough,
}) => {
  const playerError = usePlayerError();
  const { activeRestriction, handlePlayerError, removeRestriction } =
    usePlaybackRestrictions();

  useEffect(() => {
    if (playerError) {
      handlePlayerError(playerError);
    }
  }, [handlePlayerError, playerError]);

  if (passThrough) {
    return <>{children}</>;
  }
  switch (activeRestriction) {
    case 'playback-error':
      return (
        <VideoError
          focusIndex={focusIndex}
          onReloadClick={() => {
            removeRestriction('playback-error');
          }}
        />
      );
    case 'player-auth-geo':
      return <GeoBlocked focusIndex={focusIndex} />;
    case 'gql-mature':
      return (
        <MatureGate
          focusIndex={focusIndex}
          onAccept={() => {
            setMatureCookie();
            removeRestriction('gql-mature');
          }}
        />
      );
    // TODO: DRM should have a custom restricion
    // A Clip cannot be created from sub-only live or vod content
    // playback-ended should never be added as a restriction since it loops indefinitely
    // Show an unknown restriction in each of these cases
    case 'player-auth-drm':
    case 'gql-sub-only':
    case 'player-auth-vod-sub-only':
    case 'playback-ended':
    case 'player-auth-unknown':
    case 'gql-vod-muted-segments':
      return <UnknownRestriction focusIndex={0} />;
    case null:
      return <>{children}</>;
    default: {
      return exhaustedCase(
        activeRestriction,
        <UnknownRestriction focusIndex={0} />,
      );
    }
  }
};
ClipRestriction.displayName = 'ClipRestriction';
