import type { Restriction } from 'pulsar';
import {
  mockPlaybackRestrictions,
  mockPlayerController,
  setMatureCookie,
  usePlaybackRestrictions,
  usePlayerController,
} from 'pulsar';
import type { FC } from 'react';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { GeoBlocked, MatureGate, UnknownRestriction, VideoError } from '../..';
import { ClipRestriction } from '.';

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  setMatureCookie: jest.fn(),
  usePlaybackRestrictions: jest.fn(() => null),
  usePlayerController: jest.fn(() => null),
}));

const mockSetMatureCookie = setMatureCookie as jest.Mock;
const mockUsePlayerController = usePlayerController as jest.Mock;
const mockUsePlaybackRestrictions = usePlaybackRestrictions as jest.Mock;

describe(ClipRestriction, () => {
  const setup = createShallowWrapperFactory(ClipRestriction, () => ({
    clip: {
      ' $refType': 'ClipRestriction_clip',
      broadcaster: {
        broadcastSettings: {
          isMature: false,
        },
        id: validId(randomId()),
      },
    },
    focusIndex: 0,
    onErrorReloadClick: jest.fn(),
    onMatureAccept: jest.fn(),
    restriction: 'player-auth-geo',
  }));

  const restrictionComponentMappings: {
    [key in Restriction]: FC<any>;
  } = {
    'gql-mature': MatureGate,
    'gql-sub-only': UnknownRestriction,
    'gql-vod-muted-segments': UnknownRestriction,
    'playback-ended': UnknownRestriction,
    'playback-error': VideoError,
    'player-auth-drm': UnknownRestriction,
    'player-auth-geo': GeoBlocked,
    'player-auth-unknown': UnknownRestriction,
    'player-auth-vod-sub-only': UnknownRestriction,
  };

  it.each(Object.entries(restrictionComponentMappings))(
    'renders the right overlay for %s',
    (restriction, overlay) => {
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: restriction as Restriction,
        }),
      );
      const { wrapper } = setup();
      expect(wrapper.is(overlay)).toBe(true);
    },
  );

  it('renders children when there is no restriction', () => {
    const MockChildren = () => <div />;
    mockUsePlaybackRestrictions.mockReturnValue(
      mockPlaybackRestrictions({ activeRestriction: null }),
    );

    const { wrapper } = setup({ children: <MockChildren /> });

    expect(wrapper.find(MockChildren)).toExist();
  });

  describe(MatureGate, () => {
    beforeEach(() => {
      mockSetMatureCookie.mockReset();
    });

    it('sets the mature cookie on acceptance', () => {
      const mockedPlayerController = mockPlayerController({
        setMuted: jest.fn(),
      });
      mockUsePlayerController.mockReturnValue(mockedPlayerController);
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: 'gql-mature',
        }),
      );

      const { wrapper } = setup();

      expect(mockSetMatureCookie).toHaveBeenCalledTimes(0);

      wrapper.find(MatureGate).props().onAccept();

      expect(mockSetMatureCookie).toHaveBeenCalledTimes(1);
    });
  });

  describe(VideoError, () => {
    it('removes playback error when reload is clicked', () => {
      const mockRemoveRestriction = jest.fn();
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: 'playback-error',
          removeRestriction: mockRemoveRestriction,
        }),
      );

      const { wrapper } = setup();

      expect(mockRemoveRestriction).toHaveBeenCalledTimes(0);

      wrapper.find(VideoError).props().onReloadClick();

      expect(mockRemoveRestriction).toHaveBeenCalledTimes(1);
      expect(mockRemoveRestriction).toHaveBeenCalledWith('playback-error');
    });
  });

  describe('passThrough', () => {
    it('does not render the restriction when passThrough is true', () => {
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({ activeRestriction: 'gql-mature' }),
      );
      const { wrapper } = setup({
        passThrough: true,
      });

      expect(wrapper.find(MatureGate)).not.toExist();
    });
  });
});
