import type { Restriction } from 'pulsar';
import type { FC, ReactNode } from 'react';
import type { FormatMessage } from 'tachyon-intl';
import { useIntl } from 'tachyon-intl';
import {
  GlowConcernedMonitor,
  GlowGeoBlockedSvg,
  GlowMatureGateSvg,
  GlowSubOnlySvg,
} from 'tachyon-more-ui';
import {
  AlignItems,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  TextType,
} from 'twitch-core-ui';

type ErrorAssets = {
  svg: ReactNode;
  text: string;
  title: string;
};

// istanbul ignore next: trivial
const getErrorAssets = (
  restriction: Restriction,
  formatMessage: FormatMessage,
): ErrorAssets => {
  switch (restriction) {
    case 'player-auth-geo':
      return {
        svg: <GlowGeoBlockedSvg />,
        text: formatMessage(
          'This content is not available from your location.',
          'ErrorOverlay',
        ),
        title: formatMessage('Oh bummer...', 'ErrorOverlay'),
      };
    case 'gql-sub-only':
    case 'player-auth-vod-sub-only':
      return {
        svg: <GlowSubOnlySvg />,
        text: formatMessage(
          'This video is only available to subscribers. Please subscribe on Twitch.tv or your mobile device.',
          'ErrorOverlay',
        ),
        title: formatMessage('Subscribers Only', 'ErrorOverlay'),
      };
    case 'gql-mature':
      return {
        svg: <GlowMatureGateSvg />,
        text: formatMessage(
          'This channel is intended for mature audiences. Please go to the channel and accept.',
          'ErrorOverlay',
        ),
        title: formatMessage('Just one second...', 'ErrorOverlay'),
      };
    default:
      return {
        svg: <GlowConcernedMonitor />,
        text: formatMessage(
          'Looks like something went wrong. Please try again at another time.',
          'ErrorOverlay',
        ),
        title: formatMessage('Uh oh...', 'ErrorOverlay'),
      };
  }
};

/**
 * Component that renders a generic error message over the stream preview
 */
export const PreviewErrorOverlay: FC<{ restriction: Restriction }> = ({
  restriction,
}) => {
  const { formatMessage } = useIntl();
  const { svg, text, title } = getErrorAssets(restriction, formatMessage);

  return (
    <Layout
      alignItems={AlignItems.Start}
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      fullHeight
      fullWidth
      justifyContent={JustifyContent.Between}
      padding={2}
    >
      <Layout width="7.5rem">{svg}</Layout>
      <Layout>
        <Layout margin={{ bottom: 0.5 }}>
          <CoreText type={TextType.H4}>{title}</CoreText>
        </Layout>
        <CoreText>{text}</CoreText>
      </Layout>
    </Layout>
  );
};
PreviewErrorOverlay.displayName = 'PreviewErrorOverlay';
