import {
  PlayerState,
  setMatureCookie,
  usePlaybackRestrictions,
  usePlaybackState,
  usePlayerError,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect } from 'react';
import { graphql } from 'react-relay/hooks';
import type { OmitRefType } from 'tachyon-type-library';
import { exhaustedCase } from 'tachyon-utils';
import {
  GeoBlocked,
  MatureGate,
  SubOnlyVideo,
  UnknownRestriction,
  VideoError,
} from '../Overlays';
import type { BaseRestrictionProps } from '../types';
import { EndOfVodPrompt } from './EndOfVodPrompt';
import type { VodRestriction_video } from './__generated__/VodRestriction_video.graphql';

export * from './EndOfVodPrompt';

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment VodRestriction_video on Video {
    muteInfo {
      mutedSegmentConnection {
        nodes {
          duration
        }
      }
    }
    owner {
      id
      login
      broadcastSettings {
        isMature
      }
      subscriptionProducts {
        displayName
        hasSubonlyVideoArchive
      }
    }
    resourceRestriction {
      type
    }
    self {
      isRestricted
    }
    ...SubOnlyVideo_video @relay(mask: false)
  }
`;

export type VodRestrictionProps = BaseRestrictionProps & {
  onMutedSegmentsDetected?: () => void;
  video: OmitRefType<VodRestriction_video>;
};

/**
 * Component used for consistent presentation restrictions for vod content.
 * Requires `<PlayerControllerRoot>` and `<VodPlaybackRestrictionsRoot>` context providers.
 *
 * @example
 * <PlayerControllerRoot>
 *   <VodPlaybackRestrictionsRoot>
 *     <VodRestriction {...props}>
 *       {vodPlayer}
 *     </VodRestriction>
 *   </VodPlaybackRestrictionsRoot>
 * </PlayerControllerRoot>
 */
export const VodRestriction: FC<VodRestrictionProps> = ({
  children,
  focusIndex,
  onMutedSegmentsDetected,
  passThrough,
  video,
}) => {
  const playerError = usePlayerError();
  const playbackState = usePlaybackState();
  const {
    activeRestriction,
    addRestriction,
    handlePlayerError,
    removeRestriction,
  } = usePlaybackRestrictions();

  useEffect(() => {
    if (playerError) {
      handlePlayerError(playerError);
    }
  }, [handlePlayerError, playerError]);

  useEffect(() => {
    if (playbackState === PlayerState.ENDED) {
      addRestriction('playback-ended');
    }
  }, [addRestriction, playbackState]);

  useEffect(() => {
    if (activeRestriction === 'gql-vod-muted-segments') {
      onMutedSegmentsDetected?.();
      removeRestriction('gql-vod-muted-segments');
    }
  }, [activeRestriction, onMutedSegmentsDetected, removeRestriction]);

  if (passThrough) {
    return <>{children}</>;
  }
  switch (activeRestriction) {
    case 'playback-error':
      return (
        <VideoError
          focusIndex={focusIndex}
          onReloadClick={() => {
            removeRestriction('playback-error');
          }}
        />
      );
    case 'player-auth-geo':
      return <GeoBlocked focusIndex={focusIndex} />;
    case 'gql-mature':
      return (
        <MatureGate
          focusIndex={focusIndex}
          onAccept={() => {
            setMatureCookie();
            removeRestriction('gql-mature');
          }}
        />
      );
    case 'gql-sub-only':
    case 'player-auth-vod-sub-only':
      return <SubOnlyVideo focusIndex={0} video={video} />;
    // TODO: render a restriction that explains DRM
    case 'player-auth-drm':
    case 'player-auth-unknown':
      return <UnknownRestriction focusIndex={0} />;
    case 'playback-ended':
      return (
        <EndOfVodPrompt
          broadcasterLogin={video.owner?.login}
          focusIndex={focusIndex}
          onWatchAgain={() => {
            removeRestriction('playback-ended');
          }}
        />
      );
    // We handle muted segments restrictions above so
    // we can safely return children here
    case 'gql-vod-muted-segments':
    case null: {
      return <>{children}</>;
    }
    default: {
      return exhaustedCase(
        activeRestriction,
        <UnknownRestriction focusIndex={0} />,
      );
    }
  }
};
VodRestriction.displayName = 'VodRestriction';
