import { internet } from 'faker';
import type { Restriction } from 'pulsar';
import {
  mockPlaybackRestrictions,
  mockPlayerController,
  setMatureCookie,
  usePlaybackRestrictions,
  usePlayerController,
} from 'pulsar';
import type { FC } from 'react';
import { validId } from 'tachyon-relay';
import { createMountWrapperFactory, randomId } from 'tachyon-test-utils';
import {
  GeoBlocked,
  MatureGate,
  SubOnlyVideo,
  UnknownRestriction,
  VideoError,
} from '../Overlays';
import { EndOfVodPrompt, VodRestriction } from '.';

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  setMatureCookie: jest.fn(),
  usePlaybackRestrictions: jest.fn(() => null),
  usePlayerController: jest.fn(() => null),
}));

const mockSetMatureCookie = setMatureCookie as jest.Mock;
const mockUsePlayerController = usePlayerController as jest.Mock;
const mockUsePlaybackRestrictions = usePlaybackRestrictions as jest.Mock;

describe(VodRestriction, () => {
  const setup = createMountWrapperFactory(VodRestriction, () => ({
    focusIndex: 0,
    onErrorReloadClick: jest.fn(),
    onMatureAccept: jest.fn(),
    onMutedSegmentsDetected: jest.fn(),
    onWatchAgain: jest.fn(),
    restriction: 'player-auth-geo',
    video: {
      ' $refType': 'VodRestriction_video',
      muteInfo: {
        mutedSegmentConnection: {
          nodes: [],
        },
      },
      owner: {
        broadcastSettings: {
          isMature: false,
        },
        displayName: internet.userName(),
        id: validId(randomId()),
        login: internet.userName(),
        subscriptionProducts: null,
      },
      resourceRestriction: null,
      self: {
        isRestricted: false,
      },
    },
  }));

  const MockChildren = () => <div />;

  const restrictionComponentMappings: {
    [key in Restriction]: FC<any>;
  } = {
    'gql-mature': MatureGate,
    'gql-sub-only': SubOnlyVideo,
    'gql-vod-muted-segments': MockChildren,
    'playback-ended': EndOfVodPrompt,
    'playback-error': VideoError,
    'player-auth-drm': UnknownRestriction,
    'player-auth-geo': GeoBlocked,
    'player-auth-unknown': UnknownRestriction,
    'player-auth-vod-sub-only': SubOnlyVideo,
  };

  it.each(Object.entries(restrictionComponentMappings))(
    'renders the right overlay for %s',
    (restriction, overlay) => {
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: restriction as Restriction,
        }),
      );
      const props =
        restriction === 'gql-vod-muted-segments'
          ? { children: <MockChildren /> }
          : {};
      const { wrapper } = setup(props);
      expect(wrapper.find(overlay)).toExist();
    },
  );

  describe(MatureGate, () => {
    beforeEach(() => {
      mockSetMatureCookie.mockReset();
    });

    it('sets the mature cookie on acceptance', () => {
      const mockedPlayerController = mockPlayerController({
        setMuted: jest.fn(),
      });
      mockUsePlayerController.mockReturnValue(mockedPlayerController);
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({ activeRestriction: 'gql-mature' }),
      );

      const { wrapper } = setup();

      expect(mockSetMatureCookie).toHaveBeenCalledTimes(0);

      wrapper.find(MatureGate).props().onAccept();

      expect(mockSetMatureCookie).toHaveBeenCalledTimes(1);
    });
  });

  describe(VideoError, () => {
    it('removes playback error when reload is clicked', () => {
      const mockRemoveRestriction = jest.fn();
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: 'playback-error',
          removeRestriction: mockRemoveRestriction,
        }),
      );

      const { wrapper } = setup();

      expect(mockRemoveRestriction).toHaveBeenCalledTimes(0);

      wrapper.find(VideoError).props().onReloadClick();

      expect(mockRemoveRestriction).toHaveBeenCalledTimes(1);
      expect(mockRemoveRestriction).toHaveBeenCalledWith('playback-error');
    });
  });

  describe(EndOfVodPrompt, () => {
    it('removes playback ended when watch again is clicked', () => {
      const mockRemoveRestriction = jest.fn();
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: 'playback-ended',
          removeRestriction: mockRemoveRestriction,
        }),
      );

      const { wrapper } = setup();

      expect(mockRemoveRestriction).toHaveBeenCalledTimes(0);

      wrapper.find(EndOfVodPrompt).props().onWatchAgain();

      expect(mockRemoveRestriction).toHaveBeenCalledTimes(1);
      expect(mockRemoveRestriction).toHaveBeenCalledWith('playback-ended');
    });
  });

  describe('passThrough', () => {
    it('does not render the restriction when passThrough is true', () => {
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({ activeRestriction: 'gql-mature' }),
      );
      const { wrapper } = setup({
        passThrough: true,
      });

      expect(wrapper.find(MatureGate)).not.toExist();
    });

    it('renders muted segments when passThrough is true', () => {
      const mockOnMutedSegmentsDetected = jest.fn();
      mockUsePlaybackRestrictions.mockReturnValue(
        mockPlaybackRestrictions({
          activeRestriction: 'gql-vod-muted-segments',
        }),
      );
      const { wrapper } = setup({
        onMutedSegmentsDetected: mockOnMutedSegmentsDetected,
        passThrough: true,
      });

      wrapper.update();

      expect(mockOnMutedSegmentsDetected).toHaveBeenCalled();
    });
  });
});
