import {
  PlayerType,
  createStreamManifestUrl,
  useStreamAnalytics,
} from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { useEffectOnce } from 'tachyon-utils';
import { getPulsarChannelOrVodBackend } from '../../../config';
import { PlayerLoadingOverlay } from '../PlayerLoadingOverlay';
import type { PulsarBackendLoaderProps } from '../PulsarBackendLoader';
import { PulsarBackendLoader } from '../PulsarBackendLoader';
import { useDiscoveryTrackingEvent } from '../useDiscoveryTrackingEvent';
import { useRefetchPlaybackAccessToken } from '../useRefetchPlaybackAccessToken';
import type { StreamPlayer_channel$key } from './__generated__/StreamPlayer_channel.graphql';
import type { StreamPlayer_currentUser$key } from './__generated__/StreamPlayer_currentUser.graphql';
import type { StreamPlayer_token$key } from './__generated__/StreamPlayer_token.graphql';

export type {
  StreamPlayer_token$key,
  StreamPlayer_token,
} from './__generated__/StreamPlayer_token.graphql';

export type StreamPlayerProps = Omit<
  PulsarBackendLoaderProps,
  'backend' | 'src'
> & {
  channel: StreamPlayer_channel$key;
  currentUser: StreamPlayer_currentUser$key | null;
  isPreviewPlayer?: boolean;
  onManifestUrlCreated?: (url: string) => void;
  streamToken: StreamPlayer_token$key;
};

export const streamTokenFragment = graphql`
  fragment StreamPlayer_token on Query
  @refetchable(queryName: "StreamPlayer_AccessToken") {
    user(login: $login) {
      login
      stream @skip(if: $skipPlayToken) {
        playbackAccessToken(
          params: { platform: $platform, playerType: $playerType }
        ) {
          signature
          value
        }
      }
    }
  }
`;

const streamTokenRefetchQuery = graphql`
  query StreamPlayer_Query(
    $login: String!
    $playerType: String!
    $platform: String!
    $skipPlayToken: Boolean!
  ) {
    ...StreamPlayer_token
  }
`;

const streamPlayerChannelFragment = graphql`
  fragment StreamPlayer_channel on User {
    id
    login
    roles {
      isPartner
    }
    self {
      subscriptionBenefit {
        id
      }
    }
    stream {
      id
      game {
        name
      }
      previewImageURL
    }
  }
`;

const streamPlayerCurrentUserFragment = graphql`
  fragment StreamPlayer_currentUser on User {
    hasTurbo
    id
  }
`;

export const StreamPlayer: FC<StreamPlayerProps> = ({
  channel: channelRef,
  currentUser: currentUserRef,
  isPreviewPlayer,
  onManifestUrlCreated,
  streamToken: streamTokenRef,
  ...playerProps
}) => {
  const {
    common: { platform },
  } = useStaticEnvironment();
  const currentUser = useFragment(
    streamPlayerCurrentUserFragment,
    currentUserRef,
  );
  const channel = useFragment(streamPlayerChannelFragment, channelRef);

  const playerType = isPreviewPlayer
    ? PlayerType.Max480pNoAds
    : PlayerType.Pulsar;

  const { refetchToken, token: tokenResponse } = useRefetchPlaybackAccessToken({
    login: channel.login,
    playerType,
    refetchQuery: streamTokenRefetchQuery,
    tokenFragment: streamTokenFragment,
    tokenRef: streamTokenRef,
  });

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useDiscoveryTrackingEvent();
  useStreamAnalytics({
    channel,
    currentUser,
    onTrackingEvent,
    tracking: {
      playerType,
      sessionId: sessionID,
    },
  });

  useEffectOnce(() => {
    if (!tokenResponse.user?.stream) {
      refetchToken();
    }
  });

  const manifestUrl = createStreamManifestUrl(tokenResponse.user, {
    adResolutionMatching: platform === Platform.Switch,
    playerType,
  });

  if (manifestUrl) {
    onManifestUrlCreated?.(manifestUrl);
  }

  return (
    <PlayerLoadingOverlay>
      <PulsarBackendLoader
        backend={getPulsarChannelOrVodBackend(platform)}
        src={manifestUrl}
        {...playerProps}
      />
    </PlayerLoadingOverlay>
  );
};

StreamPlayer.displayName = 'StreamPlayer';
