import { datatype, internet } from 'faker';
import {
  PlayerType,
  createStreamManifestUrl,
  mockPlayerController,
  usePlayerController,
} from 'pulsar';
import { useFragment } from 'react-relay/hooks';
import {
  Platform,
  createMockStaticEnvironmentContext,
  getOtherEnvMock,
} from 'tachyon-environment';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { OmitRefType } from 'tachyon-type-library';
import { PulsarBackendLoader } from '../PulsarBackendLoader';
import { useRefetchPlaybackAccessToken } from '../useRefetchPlaybackAccessToken';
import type { StreamPlayer_channel } from './__generated__/StreamPlayer_channel.graphql';
import { StreamPlayer } from '.';
import type { StreamPlayer_token } from '.';

jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useFragment: jest.fn(),
}));
const mockUseFragment = useFragment as jest.Mock;

jest.mock('../PulsarBackendLoader', () => ({
  PulsarBackendLoader: jest.fn(() => <div />),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerController: jest.fn(),
}));

jest.mock('../useRefetchPlaybackAccessToken', () => ({
  useRefetchPlaybackAccessToken: jest.fn(),
}));
const mockUseRefetchPlaybackAccessToken =
  useRefetchPlaybackAccessToken as jest.Mock;

function mockFragments(
  overrides: Partial<StreamPlayer_channel> = {},
): [
  OmitRefType<StreamPlayer_channel>,
  { token: OmitRefType<StreamPlayer_token> },
] {
  const login = internet.userName();

  return [
    {
      id: datatype.uuid(),
      login,
      roles: {
        isPartner: datatype.boolean(),
      },
      self: null,
      stream: {
        game: null,
        id: datatype.uuid(),
        previewImageURL: null,
      },
      ...overrides,
    },
    {
      token: {
        user: {
          login,
          stream: {
            playbackAccessToken: {
              signature: '123',
              value: 'some-token',
            },
          },
        },
      },
    },
  ];
}

describe(StreamPlayer, () => {
  const setup = createMountWrapperFactory(StreamPlayer, () => ({
    channel: { ' $fragmentRefs': { StreamPlayer_channel: true } },
    currentUser: null,
    onManifestUrlCreated: jest.fn(),
    streamToken: { ' $fragmentRefs': { StreamPlayer_token: true } },
  }));

  const mockedPlayerController = mockPlayerController();

  beforeEach(() => {
    mockUseFragment.mockReset();
    mockUseRefetchPlaybackAccessToken.mockReset();
    // Mock the first call, used for `currentUser`
    mockUseFragment.mockImplementationOnce(() => null);
    (usePlayerController as jest.Mock).mockImplementation(
      () => mockedPlayerController,
    );
  });

  describe('preview player', () => {
    it('sets the Max480pNoAds player type for the manifest URL', () => {
      const [channel, tokenResponse] = mockFragments();
      mockUseFragment.mockImplementationOnce(() => channel);
      mockUseRefetchPlaybackAccessToken.mockImplementationOnce(
        () => tokenResponse,
      );

      const { props, wrapper } = setup({ isPreviewPlayer: true });

      expect(props.onManifestUrlCreated).toHaveBeenCalledTimes(1);
      expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
        src: createStreamManifestUrl(tokenResponse.token.user, {
          playerType: PlayerType.Max480pNoAds,
        }),
      });
    });
  });

  describe('Nintendo Switch', () => {
    it('enables ad resolution matching for the manifest URL', () => {
      const [channel, tokenResponse] = mockFragments();
      mockUseFragment.mockImplementationOnce(() => channel);
      mockUseRefetchPlaybackAccessToken.mockImplementationOnce(
        () => tokenResponse,
      );

      const { props, wrapper } = setup(
        {},
        {
          wrappingContexts: [
            createMockStaticEnvironmentContext(
              getOtherEnvMock({ platform: Platform.Switch }),
            ),
          ],
        },
      );

      expect(props.onManifestUrlCreated).toHaveBeenCalledTimes(1);
      expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
        src: createStreamManifestUrl(tokenResponse.token.user, {
          adResolutionMatching: true,
          playerType: PlayerType.Pulsar,
        }),
      });
    });
  });

  it('renders a PulsarBackendLoader with the expected params', () => {
    const [channel, tokenResponse] = mockFragments();
    mockUseFragment.mockImplementationOnce(() => channel);
    mockUseRefetchPlaybackAccessToken.mockImplementationOnce(
      () => tokenResponse,
    );

    const { props, wrapper } = setup();

    expect(props.onManifestUrlCreated).toHaveBeenCalledTimes(1);
    expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
      src: createStreamManifestUrl(tokenResponse.token.user, {
        playerType: PlayerType.Pulsar,
      }),
    });
  });
});
