import { ChannelPlaybackRestrictionsRoot, PlayerControllerRoot } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import type { StreamType } from 'tachyon-more-ui';
import {
  ClientOnly,
  getChannelThumbnailSrcAndSrcSet,
  getStreamType,
} from 'tachyon-more-ui';
import { getPlayingChannel } from 'tachyon-page-utils';
import { CoreImage } from 'twitch-core-ui';
import { PreviewRestriction } from '../Restrictions';
import type { StreamPlayerProps } from '../StreamPlayer';
import { StreamPlayer } from '../StreamPlayer';
import { StreamPlayerOverlay } from './StreamPlayerOverlay';
import type { StreamPreviewPlayer_channel } from './__generated__/StreamPreviewPlayer_channel.graphql';
import type { StreamPreviewPlayer_currentUser } from './__generated__/StreamPreviewPlayer_currentUser.graphql';

export type StreamPreviewPlayerProps = Pick<
  StreamPlayerProps,
  'streamToken'
> & {
  channel: StreamPreviewPlayer_channel;
  currentUser: StreamPreviewPlayer_currentUser | null;
};

/**
 * Component intended to be embedded within a page that plays a stream.
 * Will always be muted, ignores captions, and ignores content restrictions.
 */
export const StreamPreviewPlayerBase: FC<StreamPreviewPlayerProps> = ({
  channel,
  currentUser,
  streamToken,
}) => {
  const {
    common: { platform },
  } = useStaticEnvironment();

  const playingChannel = getPlayingChannel(channel);
  let streamType: StreamType | undefined;
  if (playingChannel) {
    streamType = getStreamType({
      broadcaster: channel,
      type: playingChannel.stream.type,
    });
  }

  if (!playingChannel || !streamType) {
    return null;
  }

  // Playback on Switch results in fullscreen take over by a native UI, so show a preview image instead
  if (platform === Platform.Switch) {
    return (
      <CoreImage
        alt=""
        {...getChannelThumbnailSrcAndSrcSet(channel.stream?.previewImageURL)}
      />
    );
  }

  return (
    <PreviewRestriction>
      <ClientOnly>
        <StreamPlayer
          autoPlay
          channel={playingChannel}
          currentUser={currentUser}
          isPreviewPlayer
          muted
          streamToken={streamToken}
        />
      </ClientOnly>
      <StreamPlayerOverlay
        streamType={streamType}
        viewersCount={playingChannel.stream.viewersCount ?? 0}
      />
    </PreviewRestriction>
  );
};
StreamPreviewPlayerBase.displayName = 'StreamPreviewPlayerBase';

// Consistent fragment for structure of the top level user and hosted user
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment StreamPreviewPlayer_channelBase on User {
    ...StreamPlayer_channel
    id
    login
    stream {
      id
      type
      viewersCount
      previewImageURL
    }
  }
`;

// istanbul ignore next: trivial
export const StreamPreviewPlayer = createFragmentContainer(
  (props: StreamPreviewPlayerProps) => (
    <PlayerControllerRoot>
      <ChannelPlaybackRestrictionsRoot channel={props.channel}>
        <StreamPreviewPlayerBase {...props} />
      </ChannelPlaybackRestrictionsRoot>
    </PlayerControllerRoot>
  ),
  {
    channel: graphql`
      fragment StreamPreviewPlayer_channel on User {
        hosting {
          ...StreamPreviewPlayer_channelBase @relay(mask: false)
        }
        ...StreamPreviewPlayer_channelBase @relay(mask: false)
        ...ChannelRestriction_channel @relay(mask: false)
      }
    `,
    currentUser: graphql`
      fragment StreamPreviewPlayer_currentUser on User {
        ...StreamPlayer_currentUser
      }
    `,
  },
);

StreamPreviewPlayer.displayName = 'StreamPreviewPlayer';
