import { datatype, image, internet } from 'faker';
import { usePlayerError } from 'pulsar';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { StreamType } from 'tachyon-more-ui';
import { validId } from 'tachyon-relay';
import { createShallowWrapperFactory, randomId } from 'tachyon-test-utils';
import { CoreImage } from 'twitch-core-ui';
import { PreviewErrorOverlay } from '../Restrictions';
import { StreamPlayer } from '../StreamPlayer';
import { StreamPlayerOverlay } from './StreamPlayerOverlay';
import { StreamPreviewPlayerBase } from '.';

const channelWithBadStreamType = () => {
  return {
    stream: {
      id: validId(randomId()),
      type: undefined,
    },
  };
};

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerError: jest.fn(),
}));
jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));

const mockUsePlayerError = usePlayerError as jest.Mock;
const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;

describe(StreamPreviewPlayerBase, () => {
  const setup = createShallowWrapperFactory(StreamPreviewPlayerBase, () => ({
    channel: {
      ' $fragmentRefs': {
        ChannelRestriction_channel: true,
        StreamPlayer_channel: true,
      },
      ' $refType': 'StreamPreviewPlayer_channel',
      broadcastSettings: {
        isMature: false,
      },
      displayName: internet.userName(),
      hosting: null,
      id: validId(randomId()),
      login: internet.userName(),
      stream: {
        id: validId(randomId()),
        previewImageURL: image.imageUrl(),
        restrictionType: '',
        self: {
          canWatch: true,
        },
        type: StreamType.Live,
        viewersCount: datatype.number(),
      },
    },
    currentUser: null,
    streamToken: { ' $fragmentRefs': { StreamPlayer_token: true } },
  }));

  beforeEach(() => {
    mockUsePlayerError.mockReset();
    mockUsePlayerError.mockReturnValue(null);
    mockUseStaticEnvironment.mockReturnValue({
      common: { platform: Platform.StarshotDev },
    });
  });

  it('renders nothing if there is no stream type', () => {
    const { wrapper } = setup({ channel: channelWithBadStreamType() });

    expect(wrapper.find(StreamPlayer)).not.toExist();
    expect(wrapper.find(PreviewErrorOverlay)).not.toExist();
    expect(wrapper.find(StreamPlayerOverlay)).not.toExist();
  });

  it('displays the player overlay if there is a playing channel', () => {
    const { wrapper } = setup();

    expect(wrapper.find(StreamPlayer)).toExist();
    expect(wrapper.find(StreamPlayerOverlay)).toExist();
  });

  it('displays the player overlay if there is a hosted playing channel', () => {
    const { props, wrapper } = setup({
      channel: {
        hosting: {
          id: validId(randomId()),
          login: internet.userName(),
          stream: {
            id: validId(randomId()),
            type: StreamType.Live,
            viewersCount: datatype.number(),
          },
        },
        stream: null,
      },
    });

    expect(wrapper.find(StreamPlayer).prop('channel')).toMatchObject({
      login: props.channel.hosting!.login,
    });
    expect(wrapper.find(StreamPlayerOverlay)).toExist();
  });

  it('displays a preview image on Switch', () => {
    mockUseStaticEnvironment.mockReturnValueOnce({
      common: { platform: Platform.Switch },
    });
    const { wrapper } = setup();

    expect(wrapper.find(StreamPlayer)).not.toExist();
    expect(wrapper.find(StreamPlayerOverlay)).not.toExist();
    expect(wrapper.find(CoreImage)).toExist();
  });
});
