import type { PlaybackAccessToken, PlayerType } from 'pulsar';
import { createVideoManifestUrl, useVodAnalytics } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { useStartTime } from 'tachyon-page-utils';
import { convertToUnsafeID } from 'tachyon-relay';
import {
  PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY,
  getPulsarChannelOrVodBackend,
} from '../../../../config';
import { PlayerLoadingOverlay } from '../../PlayerLoadingOverlay';
import type { PulsarBackendLoaderProps } from '../../PulsarBackendLoader';
import { PulsarBackendLoader } from '../../PulsarBackendLoader';
import { useDiscoveryTrackingEvent } from '../../useDiscoveryTrackingEvent';
import type { VodPlayerBase_currentUser$key } from './__generated__/VodPlayerBase_currentUser.graphql';
import type { VodPlayerBase_video$key } from './__generated__/VodPlayerBase_video.graphql';

export const vodPlayerVideoFragment = graphql`
  fragment VodPlayerBase_video on Video {
    broadcastType
    id
    game {
      name
    }
    owner {
      id
      login
      roles {
        isPartner
      }
      self {
        subscriptionBenefit {
          id
        }
      }
    }
    self {
      viewingHistory {
        position
      }
    }
  }
`;

const vodPlayerCurrentUserFragment = graphql`
  fragment VodPlayerBase_currentUser on User {
    id
    hasTurbo
  }
`;

export type VodPlayerBaseProps = Omit<
  PulsarBackendLoaderProps,
  'backend' | 'src' | 'startTime'
> & {
  currentUser: VodPlayerBase_currentUser$key | null;
  onManifestUrlCreated?: (url: string) => void;
  playbackAccessToken: PlaybackAccessToken | null;
  playerType: PlayerType;
  video: VodPlayerBase_video$key;
};

export const VodPlayerBase: FC<VodPlayerBaseProps> = ({
  currentUser: currentUserRef,
  onManifestUrlCreated,
  playbackAccessToken,
  playerType,
  video: videoRef,
  ...playerProps
}) => {
  const currentUser = useFragment(vodPlayerCurrentUserFragment, currentUserRef);
  const video = useFragment(vodPlayerVideoFragment, videoRef);
  const {
    client: { sessionID },
    common: { platform },
  } = useStaticEnvironment();

  const startTime = useStartTime(
    PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY,
    video.self,
  );

  const onTrackingEvent = useDiscoveryTrackingEvent();
  useVodAnalytics({
    currentUser,
    onTrackingEvent,
    tracking: {
      playerType,
      sessionId: sessionID,
    },
    video,
  });

  const manifestUrl = createVideoManifestUrl(
    {
      id: convertToUnsafeID(video.id),
      playbackAccessToken,
    },
    {
      // Switch 12.1.0 cannot handle 720p60 Twitch VOD Segment sizes.
      // More info: https://docs.google.com/document/d/1_-N5wl171wJCUrHcig93leFsdfdgKoJcmAuUVJIHwnc/edit
      // Set the max slightly higher than 30 since it's common for values to range from 30-35
      maxFrameRate: platform === Platform.Switch ? 35 : undefined,
      maxHeight: platform === Platform.Switch ? 720 : undefined,
      playerType,
    },
  );

  if (manifestUrl) {
    onManifestUrlCreated?.(manifestUrl);
  }

  return (
    <PlayerLoadingOverlay>
      <PulsarBackendLoader
        backend={getPulsarChannelOrVodBackend(platform)}
        src={manifestUrl}
        startTime={startTime}
        {...playerProps}
      />
    </PlayerLoadingOverlay>
  );
};

VodPlayerBase.displayName = 'VodPlayerBase';
