import { datatype, internet } from 'faker';
import {
  PlayerType,
  createVideoManifestUrl,
  mockPlayerController,
  usePlayerController,
} from 'pulsar';
import { useFragment } from 'react-relay/hooks';
import {
  Platform,
  createMockStaticEnvironmentContext,
  getOtherEnvMock,
} from 'tachyon-environment';
import { mockRouterUtils, useRouterUtils } from 'tachyon-next-routing-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { PulsarBackendLoader } from '../../PulsarBackendLoader';
import type {
  VodPlayerBase_video,
  VodPlayerBase_video$key,
} from './__generated__/VodPlayerBase_video.graphql';
import { VodPlayerBase } from '.';

jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useFragment: jest.fn(),
}));
const mockUseFragment = useFragment as jest.Mock;

jest.mock('tachyon-next-routing-utils', () => ({
  ...jest.requireActual('tachyon-next-routing-utils'),
  useRouterUtils: jest.fn(),
}));
const mockUseRouterUtils = useRouterUtils as jest.Mock;

jest.mock('../../PulsarBackendLoader', () => ({
  PulsarBackendLoader: jest.fn(() => <div />),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerController: jest.fn(),
}));

function mockVideo(
  overrides: Partial<VodPlayerBase_video> = {},
): VodPlayerBase_video {
  return {
    ' $refType': 'VodPlayerBase_video',
    broadcastType: 'ARCHIVE',
    game: null,
    id: datatype.uuid(),
    owner: {
      id: datatype.uuid(),
      login: internet.userName(),
      roles: {
        isPartner: datatype.boolean(),
      },
      self: null,
    },
    self: null,
    ...overrides,
  };
}

describe(VodPlayerBase, () => {
  const setup = createMountWrapperFactory(VodPlayerBase, () => ({
    currentUser: null,
    playbackAccessToken: {
      signature: datatype.uuid(),
      value: datatype.uuid(),
    },
    playerType: PlayerType.Pulsar,
    video: {} as VodPlayerBase_video$key,
  }));

  const mockedPlayerController = mockPlayerController();

  beforeEach(() => {
    mockUseFragment.mockReset();
    // Mock the first call, used for `currentUser`
    mockUseFragment.mockImplementationOnce(() => null);
    mockUseRouterUtils.mockReturnValue(mockRouterUtils());
    (usePlayerController as jest.Mock).mockImplementation(
      () => mockedPlayerController,
    );
  });

  describe('Nintendo Switch', () => {
    it('enables ad resolution matching for the manifest URL', () => {
      const video = mockVideo();
      mockUseFragment.mockImplementationOnce(() => video);

      const {
        props: { playbackAccessToken, playerType },
        wrapper,
      } = setup(
        {},
        {
          wrappingContexts: [
            createMockStaticEnvironmentContext(
              getOtherEnvMock({ platform: Platform.Switch }),
            ),
          ],
        },
      );

      expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
        src: createVideoManifestUrl(
          {
            id: video.id,
            playbackAccessToken,
          },
          { maxFrameRate: 35, maxHeight: 720, playerType },
        ),
      });
    });
  });

  it('renders a PulsarBackendLoader with the expected params', () => {
    const video = mockVideo({ id: datatype.uuid() });
    mockUseFragment.mockImplementationOnce(() => video);

    const { props, wrapper } = setup({
      playbackAccessToken: {
        signature: datatype.uuid(),
        value: datatype.uuid(),
      },
    });

    expect(wrapper.find(PulsarBackendLoader)).toHaveProp({
      src: createVideoManifestUrl({
        id: video.id,
        playbackAccessToken: props.playbackAccessToken,
      }),
    });
  });

  it('sets the manifest url when a callback is provided', () => {
    const mockSetManifestUrl = jest.fn();
    const video = mockVideo();
    mockUseFragment.mockImplementationOnce(() => video);
    setup({
      onManifestUrlCreated: mockSetManifestUrl,
    });
    expect(mockSetManifestUrl).toHaveBeenCalledTimes(1);
  });
});
