import { PlayerType } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useEffectOnce } from 'tachyon-utils';
import { useRefetchPlaybackAccessToken } from '../../useRefetchPlaybackAccessToken';
import type { VodPlayerBaseProps } from '../VodPlayerBase';
import { VodPlayerBase, vodPlayerVideoFragment } from '../VodPlayerBase';
import type { VodPlayerWrapper_token$key } from './__generated__/VodPlayerWrapper_token.graphql';

export type {
  VodPlayerWrapper_token$key,
  VodPlayerWrapper_token,
} from './__generated__/VodPlayerWrapper_token.graphql';

const vodTokenFragment = graphql`
  fragment VodPlayerWrapper_token on Query
  @refetchable(queryName: "VodPlayerWrapper_AccessToken") {
    video(id: $videoId) @skip(if: $skipPlayToken) {
      playbackAccessToken(
        params: { platform: $platform, playerType: $playerType }
      ) {
        signature
        value
      }
    }
  }
`;

const vodTokenRefetchQuery = graphql`
  query VodPlayerWrapper_Query(
    $videoId: ID!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    ...VodPlayerWrapper_token
  }
`;

export type VodPlayerWrapperProps = Omit<
  VodPlayerBaseProps,
  'playbackAccessToken' | 'playerType'
> & {
  vodToken: VodPlayerWrapper_token$key;
};

/**
 * Wrapper responsible for refetching the playback access token from
 * the Vod playback page query.
 */
// istanbul ignore next: trivial
export const VodPlayerWrapper: FC<VodPlayerWrapperProps> = ({
  vodToken: vodTokenRef,
  ...props
}) => {
  const playerType = PlayerType.Pulsar;
  const video = useFragment(vodPlayerVideoFragment, props.video);
  const { refetchToken, token: vodToken } = useRefetchPlaybackAccessToken({
    playerType,
    refetchQuery: vodTokenRefetchQuery,
    tokenFragment: vodTokenFragment,
    tokenRef: vodTokenRef,
    videoId: video.id,
  });

  useEffectOnce(() => {
    if (!vodToken.video?.playbackAccessToken) {
      refetchToken();
    }
  });

  const playbackAccessToken = vodToken.video?.playbackAccessToken ?? null;

  return (
    <VodPlayerBase
      playbackAccessToken={playbackAccessToken}
      playerType={playerType}
      {...props}
    />
  );
};

VodPlayerWrapper.displayName = 'VodPlayerWrapper';
